@extends('layouts.admin')

@section('page-title')
    {{ __('Gantt Chart') }} - {{ $project->project_name }}
@endsection

@section('breadcrumb')
    <li class="breadcrumb-item"><a href="{{ route('dashboard') }}">{{ __('Dashboard') }}</a></li>
    <li class="breadcrumb-item"><a href="{{ route('projects.index') }}">{{ __('Projects') }}</a></li>
    <li class="breadcrumb-item"><a href="{{ route('projects.show', $project->id) }}">{{ $project->project_name }}</a></li>
    <li class="breadcrumb-item">{{ __('Gantt Chart') }}</li>
@endsection

@section('action-btn')
    <div class="float-end">
        <a href="{{ route('construction.stages.index', $project->id) }}" class="btn btn-sm btn-secondary">
            <i class="ti ti-arrow-left"></i> {{ __('Back to Stages') }}
        </a>
    </div>
@endsection

@push('css-page')
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/frappe-gantt@0.6.1/dist/frappe-gantt.min.css">
    <style>
        .gantt-container {
            overflow-x: auto;
            background: white;
            padding: 20px;
            border-radius: 5px;
        }
        .gantt .bar {
            fill: #6c757d;
        }
        .gantt .bar-progress {
            fill: #0d6efd;
        }
        .gantt-activity-pending .bar {
            fill: #6c757d;
        }
        .gantt-activity-in_progress .bar {
            fill: #0d6efd;
        }
        .gantt-activity-in_progress .bar-progress {
            fill: #0b5ed7;
        }
        .gantt-activity-completed .bar {
            fill: #198754;
        }
        .gantt-activity-completed .bar-progress {
            fill: #157347;
        }
        .gantt-activity-rework .bar {
            fill: #dc3545;
        }
        .gantt-activity-rework .bar-progress {
            fill: #b02a37;
        }
        .gantt .bar-label {
            fill: #fff;
            font-weight: 500;
        }
        .gantt-legend {
            display: flex;
            gap: 20px;
            margin-bottom: 20px;
            padding: 15px;
            background: white;
            border-radius: 5px;
        }
        .gantt-legend-item {
            display: flex;
            align-items: center;
            gap: 8px;
        }
        .gantt-legend-color {
            width: 20px;
            height: 20px;
            border-radius: 3px;
        }
        .filter-section {
            background: white;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
        }
        .filter-section .form-check {
            display: inline-block;
            margin-right: 20px;
        }
    </style>
@endpush

@section('content')
    <div class="row">
        <div class="col-md-12">
            <div class="card">
                <div class="card-header">
                    <h5>{{ __('Project Timeline Overview') }}</h5>
                </div>
                <div class="card-body">
                    <div class="gantt-legend">
                        <div class="gantt-legend-item">
                            <div class="gantt-legend-color" style="background: #6c757d;"></div>
                            <span>{{ __('Pending') }}</span>
                        </div>
                        <div class="gantt-legend-item">
                            <div class="gantt-legend-color" style="background: #0d6efd;"></div>
                            <span>{{ __('In Progress') }}</span>
                        </div>
                        <div class="gantt-legend-item">
                            <div class="gantt-legend-color" style="background: #198754;"></div>
                            <span>{{ __('Completed') }}</span>
                        </div>
                        <div class="gantt-legend-item">
                            <div class="gantt-legend-color" style="background: #dc3545;"></div>
                            <span>{{ __('Rework Required') }}</span>
                        </div>
                    </div>

                    <div class="filter-section">
                        <h6>{{ __('Filter by Status') }}</h6>
                        <div class="form-check">
                            <input class="form-check-input status-filter" type="checkbox" value="pending" id="filter-pending" checked>
                            <label class="form-check-label" for="filter-pending">{{ __('Pending') }}</label>
                        </div>
                        <div class="form-check">
                            <input class="form-check-input status-filter" type="checkbox" value="in_progress" id="filter-in_progress" checked>
                            <label class="form-check-label" for="filter-in_progress">{{ __('In Progress') }}</label>
                        </div>
                        <div class="form-check">
                            <input class="form-check-input status-filter" type="checkbox" value="completed" id="filter-completed" checked>
                            <label class="form-check-label" for="filter-completed">{{ __('Completed') }}</label>
                        </div>
                        <div class="form-check">
                            <input class="form-check-input status-filter" type="checkbox" value="rework" id="filter-rework" checked>
                            <label class="form-check-label" for="filter-rework">{{ __('Rework Required') }}</label>
                        </div>
                    </div>

                    @if(count($ganttData) > 0)
                        <div class="gantt-container">
                            <svg id="gantt"></svg>
                        </div>
                    @else
                        <div class="text-center py-5">
                            <i class="ti ti-calendar-time" style="font-size: 48px; opacity: 0.3;"></i>
                            <p class="text-muted mt-3">{{ __('No activities with dates found.') }}</p>
                            <p class="text-muted">{{ __('Add start and end dates to activities to view them in the Gantt chart.') }}</p>
                        </div>
                    @endif
                </div>
            </div>
        </div>
    </div>

    <div class="row">
        <div class="col-md-12">
            <div class="card">
                <div class="card-header">
                    <h5>{{ __('Activity Details') }}</h5>
                </div>
                <div class="card-body table-border-style">
                    @if(count($ganttData) > 0)
                        <div class="table-responsive">
                            <table class="table" id="activity-table">
                                <thead>
                                    <tr>
                                        <th>{{ __('Activity') }}</th>
                                        <th>{{ __('Stage') }}</th>
                                        <th>{{ __('Status') }}</th>
                                        <th>{{ __('Progress') }}</th>
                                        <th>{{ __('Start Date') }}</th>
                                        <th>{{ __('End Date') }}</th>
                                        <th>{{ __('Duration') }}</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    @foreach($ganttData as $item)
                                        <tr data-status="{{ $item['status'] }}">
                                            <td>{{ $item['name'] }}</td>
                                            <td>{{ $item['stage'] }}</td>
                                            <td>
                                                <span class="badge badge-{{ \App\Models\ConstructionActivity::$status_colors[$item['status']] ?? 'secondary' }}">
                                                    {{ \App\Models\ConstructionActivity::$statuses[$item['status']] ?? 'Unknown' }}
                                                </span>
                                            </td>
                                            <td>
                                                <div class="progress" style="height: 8px;">
                                                    <div class="progress-bar bg-success" role="progressbar"
                                                         style="width: {{ $item['progress'] }}%"></div>
                                                </div>
                                                <small>{{ $item['progress'] }}%</small>
                                            </td>
                                            <td>{{ \Carbon\Carbon::parse($item['start'])->format('d M Y') }}</td>
                                            <td>{{ \Carbon\Carbon::parse($item['end'])->format('d M Y') }}</td>
                                            <td>{{ \Carbon\Carbon::parse($item['start'])->diffInDays(\Carbon\Carbon::parse($item['end'])) }} {{ __('days') }}</td>
                                        </tr>
                                    @endforeach
                                </tbody>
                            </table>
                        </div>
                    @endif
                </div>
            </div>
        </div>
    </div>
@endsection

@push('script-page')
    <script src="https://cdn.jsdelivr.net/npm/frappe-gantt@0.6.1/dist/frappe-gantt.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            @if(count($ganttData) > 0)
                let allTasks = @json($ganttData);
                let ganttInstance = null;

                function renderGantt(tasks) {
                    if (tasks.length === 0) {
                        document.getElementById('gantt').innerHTML = '<text x="50%" y="50" text-anchor="middle" fill="#999">No activities match the selected filters</text>';
                        return;
                    }

                    if (ganttInstance) {
                        ganttInstance.refresh(tasks);
                    } else {
                        ganttInstance = new Gantt("#gantt", tasks, {
                            view_mode: 'Day',
                            bar_height: 30,
                            bar_corner_radius: 3,
                            arrow_curve: 5,
                            padding: 18,
                            view_modes: ['Quarter Day', 'Half Day', 'Day', 'Week', 'Month'],
                            date_format: 'YYYY-MM-DD',
                            popup_trigger: 'click',
                            custom_popup_html: function(task) {
                                const start_date = new Date(task._start);
                                const end_date = new Date(task._end);
                                const duration = Math.ceil((end_date - start_date) / (1000 * 60 * 60 * 24));

                                return `
                                    <div class="details-container">
                                        <h5>${task.name}</h5>
                                        <p><strong>Stage:</strong> ${task.stage}</p>
                                        <p><strong>Status:</strong> ${task.status.replace('_', ' ').toUpperCase()}</p>
                                        <p><strong>Progress:</strong> ${task.progress}%</p>
                                        <p><strong>Duration:</strong> ${duration} days</p>
                                        <p><strong>Start:</strong> ${start_date.toLocaleDateString()}</p>
                                        <p><strong>End:</strong> ${end_date.toLocaleDateString()}</p>
                                    </div>
                                `;
                            },
                            on_view_change: function(mode) {
                                console.log('View mode changed to:', mode);
                            }
                        });
                    }
                }

                // Initial render
                renderGantt(allTasks);

                // Filter functionality
                const statusFilters = document.querySelectorAll('.status-filter');
                statusFilters.forEach(filter => {
                    filter.addEventListener('change', function() {
                        const activeFilters = Array.from(statusFilters)
                            .filter(f => f.checked)
                            .map(f => f.value);

                        const filteredTasks = allTasks.filter(task =>
                            activeFilters.includes(task.status)
                        );

                        renderGantt(filteredTasks);

                        // Filter table rows
                        const tableRows = document.querySelectorAll('#activity-table tbody tr');
                        tableRows.forEach(row => {
                            const status = row.getAttribute('data-status');
                            if (activeFilters.includes(status)) {
                                row.style.display = '';
                            } else {
                                row.style.display = 'none';
                            }
                        });
                    });
                });

                // View mode switcher
                const viewButtons = `
                    <div class="btn-group mb-3" role="group">
                        <button type="button" class="btn btn-sm btn-outline-primary view-mode-btn" data-mode="Quarter Day">Quarter Day</button>
                        <button type="button" class="btn btn-sm btn-outline-primary view-mode-btn" data-mode="Half Day">Half Day</button>
                        <button type="button" class="btn btn-sm btn-outline-primary view-mode-btn active" data-mode="Day">Day</button>
                        <button type="button" class="btn btn-sm btn-outline-primary view-mode-btn" data-mode="Week">Week</button>
                        <button type="button" class="btn btn-sm btn-outline-primary view-mode-btn" data-mode="Month">Month</button>
                    </div>
                `;
                document.querySelector('.gantt-container').insertAdjacentHTML('beforebegin', viewButtons);

                document.querySelectorAll('.view-mode-btn').forEach(btn => {
                    btn.addEventListener('click', function() {
                        document.querySelectorAll('.view-mode-btn').forEach(b => b.classList.remove('active'));
                        this.classList.add('active');
                        ganttInstance.change_view_mode(this.getAttribute('data-mode'));
                    });
                });
            @endif
        });
    </script>
@endpush
