<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Hash;
use App\Models\User;
use App\Models\Employee;
use App\Models\Branch;
use App\Models\Department;
use App\Models\Designation;
use Spatie\Permission\Models\Role;

class EmployeeSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Get the company user
        $companyUser = User::where('type', 'company')->first();
        
        if (!$companyUser) {
            $this->command->error('No company user found. Please create a company user first.');
            return;
        }

        $creatorId = $companyUser->creatorId();

        // Get or create default branch, department, and designation
        $branch = Branch::where('created_by', $creatorId)->first();
        if (!$branch) {
            $branch = Branch::create([
                'name' => 'Main Branch',
                'created_by' => $creatorId,
            ]);
        }

        $department = Department::where('created_by', $creatorId)->first();
        if (!$department) {
            $department = Department::create([
                'name' => 'General',
                'created_by' => $creatorId,
            ]);
        }

        $designation = Designation::where('created_by', $creatorId)->first();
        if (!$designation) {
            $designation = Designation::create([
                'name' => 'Employee',
                'created_by' => $creatorId,
            ]);
        }

        // Get Employee role
        $employeeRole = Role::where('name', 'Employee')->where('created_by', $creatorId)->first();
        if (!$employeeRole) {
            $this->command->error('Employee role not found. Please run the main seeder first.');
            return;
        }

        // Indian names with diverse regional representation
        $employees = [
            // North Indian Names
            ['name' => 'Rajesh Kumar Singh', 'email' => 'rajesh.singh@company.com', 'phone' => '+91-9876543210', 'gender' => 'Male'],
            ['name' => 'Priya Sharma', 'email' => 'priya.sharma@company.com', 'phone' => '+91-9876543211', 'gender' => 'Female'],
            ['name' => 'Amit Gupta', 'email' => 'amit.gupta@company.com', 'phone' => '+91-9876543212', 'gender' => 'Male'],
            ['name' => 'Sunita Agarwal', 'email' => 'sunita.agarwal@company.com', 'phone' => '+91-9876543213', 'gender' => 'Female'],
            ['name' => 'Vikash Yadav', 'email' => 'vikash.yadav@company.com', 'phone' => '+91-9876543214', 'gender' => 'Male'],
            
            // South Indian Names
            ['name' => 'Venkatesh Iyer', 'email' => 'venkatesh.iyer@company.com', 'phone' => '+91-9876543215', 'gender' => 'Male'],
            ['name' => 'Lakshmi Nair', 'email' => 'lakshmi.nair@company.com', 'phone' => '+91-9876543216', 'gender' => 'Female'],
            ['name' => 'Suresh Reddy', 'email' => 'suresh.reddy@company.com', 'phone' => '+91-9876543217', 'gender' => 'Male'],
            ['name' => 'Meera Krishnan', 'email' => 'meera.krishnan@company.com', 'phone' => '+91-9876543218', 'gender' => 'Female'],
            ['name' => 'Ravi Chandran', 'email' => 'ravi.chandran@company.com', 'phone' => '+91-9876543219', 'gender' => 'Male'],
            ['name' => 'Divya Menon', 'email' => 'divya.menon@company.com', 'phone' => '+91-9876543220', 'gender' => 'Female'],
            ['name' => 'Karthik Subramanian', 'email' => 'karthik.subramanian@company.com', 'phone' => '+91-9876543221', 'gender' => 'Male'],
            ['name' => 'Anjali Pillai', 'email' => 'anjali.pillai@company.com', 'phone' => '+91-9876543222', 'gender' => 'Female'],
            ['name' => 'Srinivas Rao', 'email' => 'srinivas.rao@company.com', 'phone' => '+91-9876543223', 'gender' => 'Male'],
            ['name' => 'Kavitha Naidu', 'email' => 'kavitha.naidu@company.com', 'phone' => '+91-9876543224', 'gender' => 'Female'],
            
            // West Indian Names
            ['name' => 'Arjun Patel', 'email' => 'arjun.patel@company.com', 'phone' => '+91-9876543225', 'gender' => 'Male'],
            ['name' => 'Nisha Shah', 'email' => 'nisha.shah@company.com', 'phone' => '+91-9876543226', 'gender' => 'Female'],
            ['name' => 'Rohit Desai', 'email' => 'rohit.desai@company.com', 'phone' => '+91-9876543227', 'gender' => 'Male'],
            ['name' => 'Pooja Mehta', 'email' => 'pooja.mehta@company.com', 'phone' => '+91-9876543228', 'gender' => 'Female'],
            
            // East Indian Names
            ['name' => 'Subhash Chatterjee', 'email' => 'subhash.chatterjee@company.com', 'phone' => '+91-9876543229', 'gender' => 'Male'],
            ['name' => 'Ruma Banerjee', 'email' => 'ruma.banerjee@company.com', 'phone' => '+91-9876543230', 'gender' => 'Female'],
            ['name' => 'Debashish Ghosh', 'email' => 'debashish.ghosh@company.com', 'phone' => '+91-9876543231', 'gender' => 'Male'],
            ['name' => 'Anita Das', 'email' => 'anita.das@company.com', 'phone' => '+91-9876543232', 'gender' => 'Female'],
            
            // More South Indian Names (Tamil, Telugu, Kannada, Malayalam)
            ['name' => 'Murugan Selvam', 'email' => 'murugan.selvam@company.com', 'phone' => '+91-9876543233', 'gender' => 'Male'],
            ['name' => 'Kamala Devi', 'email' => 'kamala.devi@company.com', 'phone' => '+91-9876543234', 'gender' => 'Female'],
            ['name' => 'Balakrishna Murthy', 'email' => 'balakrishna.murthy@company.com', 'phone' => '+91-9876543235', 'gender' => 'Male'],
            ['name' => 'Radha Kumari', 'email' => 'radha.kumari@company.com', 'phone' => '+91-9876543236', 'gender' => 'Female'],
            ['name' => 'Gopal Krishna', 'email' => 'gopal.krishna@company.com', 'phone' => '+91-9876543237', 'gender' => 'Male'],
            ['name' => 'Vasantha Lakshmi', 'email' => 'vasantha.lakshmi@company.com', 'phone' => '+91-9876543238', 'gender' => 'Female'],
            ['name' => 'Ramesh Babu', 'email' => 'ramesh.babu@company.com', 'phone' => '+91-9876543239', 'gender' => 'Male'],
            ['name' => 'Shanti Priya', 'email' => 'shanti.priya@company.com', 'phone' => '+91-9876543240', 'gender' => 'Female'],
        ];

        $employeeIdCounter = 1;

        foreach ($employees as $empData) {
            // Create User record
            $user = User::create([
                'name' => $empData['name'],
                'email' => $empData['email'],
                'password' => Hash::make('password123'),
                'type' => 'employee',
                'lang' => 'en',
                'created_by' => $creatorId,
                'email_verified_at' => now(),
            ]);

            // Assign Employee role
            $user->assignRole($employeeRole);

            // Generate employee ID
            $employeeId = str_pad($employeeIdCounter, 4, '0', STR_PAD_LEFT);

            // Create Employee record
            Employee::create([
                'user_id' => $user->id,
                'name' => $empData['name'],
                'dob' => now()->subYears(rand(25, 45))->format('Y-m-d'),
                'gender' => $empData['gender'],
                'phone' => $empData['phone'],
                'address' => $this->generateAddress(),
                'email' => $empData['email'],
                'password' => Hash::make('password123'),
                'employee_id' => $employeeId,
                'branch_id' => $branch->id,
                'department_id' => $department->id,
                'designation_id' => $designation->id,
                'company_doj' => now()->subMonths(rand(1, 36))->format('Y-m-d'),
                'salary_type' => 1, // Monthly
                'salary' => rand(25000, 75000),
                'is_active' => 1,
                'created_by' => $creatorId,
            ]);

            $employeeIdCounter++;
        }

        $this->command->info('✅ Successfully created ' . count($employees) . ' dummy employees with Indian names!');
        $this->command->info('📧 All employees have email: [name]@company.com and password: password123');
    }

    /**
     * Generate random Indian addresses
     */
    private function generateAddress(): string
    {
        $addresses = [
            '123, MG Road, Bangalore, Karnataka - 560001',
            '456, Anna Salai, Chennai, Tamil Nadu - 600002',
            '789, Marine Drive, Mumbai, Maharashtra - 400001',
            '321, CP, New Delhi, Delhi - 110001',
            '654, Park Street, Kolkata, West Bengal - 700016',
            '987, Banjara Hills, Hyderabad, Telangana - 500034',
            '147, Residency Road, Bangalore, Karnataka - 560025',
            '258, T Nagar, Chennai, Tamil Nadu - 600017',
            '369, Andheri West, Mumbai, Maharashtra - 400058',
            '741, Connaught Place, New Delhi, Delhi - 110001',
            '852, Salt Lake, Kolkata, West Bengal - 700064',
            '963, Jubilee Hills, Hyderabad, Telangana - 500033',
            '159, Koramangala, Bangalore, Karnataka - 560034',
            '357, Adyar, Chennai, Tamil Nadu - 600020',
            '753, Powai, Mumbai, Maharashtra - 400076',
        ];

        return $addresses[array_rand($addresses)];
    }
}