<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\ProductServiceCategory;
use App\Models\ProductServiceUnit;
use App\Models\ProductService;
use App\Models\User;

class ConstructionMaterialsSeeder extends Seeder
{
    public function run()
    {
        // Get company user
        $companyUser = User::where('type', 'company')->first();

        if (!$companyUser) {
            $this->command->error('No company user found. Please create a company user first.');
            return;
        }

        $creatorId = $companyUser->creatorId();

        // Create Units
        $units = [
            'Bag' => 'Bag (50kg)',
            'Tonne' => 'Tonne',
            'Kg' => 'Kilogram',
            'Piece' => 'Piece',
            'CFT' => 'Cubic Feet',
            'SqFt' => 'Square Feet',
            'Ltr' => 'Liter',
            'Mtr' => 'Meter',
            'Bundle' => 'Bundle',
            'Sheet' => 'Sheet',
        ];

        $unitIds = [];
        foreach ($units as $name => $fullName) {
            $unit = ProductServiceUnit::firstOrCreate(
                ['name' => $name, 'created_by' => $creatorId],
                ['name' => $name, 'created_by' => $creatorId]
            );
            $unitIds[$name] = $unit->id;
        }

        // Create Categories
        $categories = [
            'Cement',
            'Sand & Aggregates',
            'Steel & TMT Bars',
            'Bricks & Blocks',
            'Wood & Plywood',
            'Tiles & Flooring',
            'Electrical Materials',
            'Plumbing Materials',
            'Paint & Finishing',
            'Glass & Windows',
            'Hardware & Fittings',
        ];

        $categoryIds = [];
        foreach ($categories as $categoryName) {
            $category = ProductServiceCategory::firstOrCreate(
                ['name' => $categoryName, 'created_by' => $creatorId],
                [
                    'name' => $categoryName,
                    'color' => '#' . str_pad(dechex(mt_rand(0, 0xFFFFFF)), 6, '0', STR_PAD_LEFT),
                    'type' => 2, // expense type
                    'created_by' => $creatorId
                ]
            );
            $categoryIds[$categoryName] = $category->id;
        }

        // Create Materials
        $materials = [
            // Cement
            [
                'name' => 'OPC 53 Grade Cement',
                'sku' => 'CEM-OPC53',
                'brand' => 'Ultratech',
                'category' => 'Cement',
                'unit' => 'Bag',
                'description' => 'Ordinary Portland Cement 53 Grade - 50kg bag'
            ],
            [
                'name' => 'OPC 43 Grade Cement',
                'sku' => 'CEM-OPC43',
                'brand' => 'ACC',
                'category' => 'Cement',
                'unit' => 'Bag',
                'description' => 'Ordinary Portland Cement 43 Grade - 50kg bag'
            ],
            [
                'name' => 'PPC Cement',
                'sku' => 'CEM-PPC',
                'brand' => 'Ambuja',
                'category' => 'Cement',
                'unit' => 'Bag',
                'description' => 'Portland Pozzolana Cement - 50kg bag'
            ],

            // Sand
            [
                'name' => 'River Sand',
                'sku' => 'SND-RIVER',
                'brand' => null,
                'category' => 'Sand & Aggregates',
                'unit' => 'Tonne',
                'description' => 'Natural river sand for construction'
            ],
            [
                'name' => 'M Sand (Manufactured Sand)',
                'sku' => 'SND-MSAND',
                'brand' => null,
                'category' => 'Sand & Aggregates',
                'unit' => 'Tonne',
                'description' => 'Crushed aggregates for construction'
            ],
            [
                'name' => '20mm Aggregate',
                'sku' => 'AGG-20MM',
                'brand' => null,
                'category' => 'Sand & Aggregates',
                'unit' => 'Tonne',
                'description' => '20mm stone aggregate'
            ],

            // Steel
            [
                'name' => 'TMT Bar Fe 500 - 8mm',
                'sku' => 'TMT-8MM',
                'brand' => 'Tata Tiscon',
                'category' => 'Steel & TMT Bars',
                'unit' => 'Kg',
                'description' => 'Thermo Mechanically Treated steel bars'
            ],
            [
                'name' => 'TMT Bar Fe 500 - 10mm',
                'sku' => 'TMT-10MM',
                'brand' => 'Tata Tiscon',
                'category' => 'Steel & TMT Bars',
                'unit' => 'Kg',
                'description' => 'Thermo Mechanically Treated steel bars'
            ],
            [
                'name' => 'TMT Bar Fe 500 - 12mm',
                'sku' => 'TMT-12MM',
                'brand' => 'Jindal Panther',
                'category' => 'Steel & TMT Bars',
                'unit' => 'Kg',
                'description' => 'Thermo Mechanically Treated steel bars'
            ],
            [
                'name' => 'TMT Bar Fe 500 - 16mm',
                'sku' => 'TMT-16MM',
                'brand' => 'JSW Steel',
                'category' => 'Steel & TMT Bars',
                'unit' => 'Kg',
                'description' => 'Thermo Mechanically Treated steel bars'
            ],

            // Bricks
            [
                'name' => 'Red Clay Brick',
                'sku' => 'BRK-RED',
                'brand' => null,
                'category' => 'Bricks & Blocks',
                'unit' => 'Piece',
                'description' => 'Standard red clay brick 9x4x3 inch'
            ],
            [
                'name' => 'Fly Ash Brick',
                'sku' => 'BRK-FLYASH',
                'brand' => null,
                'category' => 'Bricks & Blocks',
                'unit' => 'Piece',
                'description' => 'Eco-friendly fly ash brick'
            ],
            [
                'name' => 'AAC Block 4 inch',
                'sku' => 'AAC-4IN',
                'brand' => 'Ultratech Xtralite',
                'category' => 'Bricks & Blocks',
                'unit' => 'Piece',
                'description' => 'Autoclaved Aerated Concrete block'
            ],
            [
                'name' => 'AAC Block 6 inch',
                'sku' => 'AAC-6IN',
                'brand' => 'Ultratech Xtralite',
                'category' => 'Bricks & Blocks',
                'unit' => 'Piece',
                'description' => 'Autoclaved Aerated Concrete block'
            ],

            // Wood
            [
                'name' => 'Teak Wood',
                'sku' => 'WD-TEAK',
                'brand' => null,
                'category' => 'Wood & Plywood',
                'unit' => 'CFT',
                'description' => 'Premium teak wood'
            ],
            [
                'name' => 'Commercial Plywood 8mm',
                'sku' => 'PLY-8MM',
                'brand' => 'Greenply',
                'category' => 'Wood & Plywood',
                'unit' => 'Sheet',
                'description' => '8mm commercial plywood 8x4 feet'
            ],
            [
                'name' => 'Commercial Plywood 12mm',
                'sku' => 'PLY-12MM',
                'brand' => 'Century',
                'category' => 'Wood & Plywood',
                'unit' => 'Sheet',
                'description' => '12mm commercial plywood 8x4 feet'
            ],

            // Tiles
            [
                'name' => 'Vitrified Floor Tiles 2x2',
                'sku' => 'TIL-VIT-2X2',
                'brand' => 'Kajaria',
                'category' => 'Tiles & Flooring',
                'unit' => 'SqFt',
                'description' => 'Premium vitrified tiles'
            ],
            [
                'name' => 'Ceramic Wall Tiles',
                'sku' => 'TIL-CER-WALL',
                'brand' => 'Somany',
                'category' => 'Tiles & Flooring',
                'unit' => 'SqFt',
                'description' => 'Ceramic wall tiles for bathroom/kitchen'
            ],
            [
                'name' => 'Granite Flooring',
                'sku' => 'GRN-FLOOR',
                'brand' => null,
                'category' => 'Tiles & Flooring',
                'unit' => 'SqFt',
                'description' => 'Polished granite flooring'
            ],

            // Electrical
            [
                'name' => 'Electric Wire 2.5 sq mm',
                'sku' => 'ELEC-WIRE-2.5',
                'brand' => 'Polycab',
                'category' => 'Electrical Materials',
                'unit' => 'Mtr',
                'description' => 'FR electrical wire'
            ],
            [
                'name' => 'PVC Conduit Pipe 25mm',
                'sku' => 'ELEC-COND-25',
                'brand' => 'Finolex',
                'category' => 'Electrical Materials',
                'unit' => 'Mtr',
                'description' => 'PVC conduit pipe for wiring'
            ],
            [
                'name' => 'Modular Switch',
                'sku' => 'ELEC-SWITCH',
                'brand' => 'Anchor',
                'category' => 'Electrical Materials',
                'unit' => 'Piece',
                'description' => 'Modular electrical switch'
            ],

            // Plumbing
            [
                'name' => 'CPVC Pipe 1/2 inch',
                'sku' => 'PLB-CPVC-0.5',
                'brand' => 'Ashirvad',
                'category' => 'Plumbing Materials',
                'unit' => 'Mtr',
                'description' => 'Hot & cold water pipe'
            ],
            [
                'name' => 'PVC Pipe 4 inch',
                'sku' => 'PLB-PVC-4',
                'brand' => 'Supreme',
                'category' => 'Plumbing Materials',
                'unit' => 'Mtr',
                'description' => 'PVC drainage pipe'
            ],
            [
                'name' => 'Bathroom Fittings Set',
                'sku' => 'PLB-FITTINGS',
                'brand' => 'Jaquar',
                'category' => 'Plumbing Materials',
                'unit' => 'Piece',
                'description' => 'Complete bathroom fittings'
            ],

            // Paint
            [
                'name' => 'Exterior Emulsion Paint',
                'sku' => 'PNT-EXT',
                'brand' => 'Asian Paints',
                'category' => 'Paint & Finishing',
                'unit' => 'Ltr',
                'description' => 'Weather proof exterior paint'
            ],
            [
                'name' => 'Interior Emulsion Paint',
                'sku' => 'PNT-INT',
                'brand' => 'Berger',
                'category' => 'Paint & Finishing',
                'unit' => 'Ltr',
                'description' => 'Premium interior wall paint'
            ],
            [
                'name' => 'Wall Putty',
                'sku' => 'PNT-PUTTY',
                'brand' => 'Birla White',
                'category' => 'Paint & Finishing',
                'unit' => 'Kg',
                'description' => 'White wall putty for smooth finish'
            ],

            // Glass
            [
                'name' => 'Clear Glass 5mm',
                'sku' => 'GLS-CLR-5MM',
                'brand' => 'Saint Gobain',
                'category' => 'Glass & Windows',
                'unit' => 'SqFt',
                'description' => 'Clear float glass'
            ],
            [
                'name' => 'UPVC Window Frame',
                'sku' => 'WIN-UPVC',
                'brand' => 'Fenesta',
                'category' => 'Glass & Windows',
                'unit' => 'Piece',
                'description' => 'UPVC window frame with glass'
            ],

            // Hardware
            [
                'name' => 'Door Handle Set',
                'sku' => 'HRD-HANDLE',
                'brand' => 'Godrej',
                'category' => 'Hardware & Fittings',
                'unit' => 'Piece',
                'description' => 'Stainless steel door handle'
            ],
            [
                'name' => 'Door Lock Mortise',
                'sku' => 'HRD-LOCK',
                'brand' => 'Yale',
                'category' => 'Hardware & Fittings',
                'unit' => 'Piece',
                'description' => 'Mortise door lock'
            ],
            [
                'name' => 'Tower Bolt',
                'sku' => 'HRD-BOLT',
                'brand' => 'Ozone',
                'category' => 'Hardware & Fittings',
                'unit' => 'Piece',
                'description' => 'Stainless steel tower bolt'
            ],
        ];

        foreach ($materials as $materialData) {
            ProductService::create([
                'name' => $materialData['name'],
                'sku' => $materialData['sku'],
                'brand' => $materialData['brand'],
                'category_id' => $categoryIds[$materialData['category']],
                'unit_id' => $unitIds[$materialData['unit']],
                'description' => $materialData['description'],
                'purchase_price' => null,
                'quantity' => 0,
                'tax_id' => null,
                'expense_chartaccount_id' => 0,
                'created_by' => $creatorId,
            ]);
        }

        $this->command->info('✅ Construction materials seeded successfully!');
        $this->command->info('📦 Created: ' . count($categories) . ' categories, ' . count($units) . ' units, ' . count($materials) . ' materials');
    }
}
