<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use App\Models\User;
use App\Models\Project;
use App\Models\ConstructionStage;
use App\Models\ConstructionActivity;
use Carbon\Carbon;

class ConstructionDemoSeeder extends Seeder
{
    public function run()
    {
        // Get the company user (admin)
        $companyUser = User::where('type', 'company')->first();

        if (!$companyUser) {
            $this->command->error('No company user found. Please create a company user first.');
            return;
        }

        // Create realistic construction clients
        $clients = $this->createClients($companyUser);

        // Create construction projects
        $projects = $this->createProjects($companyUser, $clients);

        // Create construction stages and activities for each project
        foreach ($projects as $project) {
            $this->createStagesAndActivities($companyUser, $project);
        }

        $this->command->info('✅ Construction demo data created successfully!');
        $this->command->info('📊 Created: ' . count($clients) . ' clients, ' . count($projects) . ' projects');
    }

    private function createClients($companyUser)
    {
        $clients = [];

        $clientData = [
            [
                'name' => 'Rajesh Kumar',
                'email' => 'rajesh.kumar@email.com',
                'password' => Hash::make('password'),
                'type' => 'client',
                'created_by' => $companyUser->creatorId(),
            ],
            [
                'name' => 'Sunita Sharma',
                'email' => 'sunita.sharma@email.com',
                'password' => Hash::make('password'),
                'type' => 'client',
                'created_by' => $companyUser->creatorId(),
            ],
            [
                'name' => 'Amit Patel',
                'email' => 'amit.patel@email.com',
                'password' => Hash::make('password'),
                'type' => 'client',
                'created_by' => $companyUser->creatorId(),
            ],
            [
                'name' => 'Priya Desai',
                'email' => 'priya.desai@email.com',
                'password' => Hash::make('password'),
                'type' => 'client',
                'created_by' => $companyUser->creatorId(),
            ],
            [
                'name' => 'Green Valley Developers',
                'email' => 'info@greenvalley.com',
                'password' => Hash::make('password'),
                'type' => 'client',
                'created_by' => $companyUser->creatorId(),
            ],
        ];

        foreach ($clientData as $data) {
            // Check if client already exists
            $existing = User::where('email', $data['email'])->first();
            if (!$existing) {
                $clients[] = User::create($data);
            } else {
                $clients[] = $existing;
            }
        }

        return $clients;
    }

    private function createProjects($companyUser, $clients)
    {
        $projects = [];

        $projectData = [
            [
                'project_name' => 'Luxury Villa Construction - Palm Heights',
                'start_date' => Carbon::now()->subMonths(3)->format('Y-m-d'),
                'end_date' => Carbon::now()->addMonths(9)->format('Y-m-d'),
                'budget' => 8500000,
                'description' => '4 BHK luxury villa with modern amenities, swimming pool, and landscaped garden. Total built-up area: 3500 sq ft.',
                'status' => 'in_progress',
                'client_id' => $clients[0]->id,
            ],
            [
                'project_name' => 'Residential Apartment - Sunrise Towers',
                'start_date' => Carbon::now()->subMonths(6)->format('Y-m-d'),
                'end_date' => Carbon::now()->addMonths(6)->format('Y-m-d'),
                'budget' => 4200000,
                'description' => '3 BHK apartment renovation and interior work. Area: 1800 sq ft. Premium finishes.',
                'status' => 'in_progress',
                'client_id' => $clients[1]->id,
            ],
            [
                'project_name' => 'Commercial Building - Tech Park Plaza',
                'start_date' => Carbon::now()->subMonths(8)->format('Y-m-d'),
                'end_date' => Carbon::now()->addMonths(16)->format('Y-m-d'),
                'budget' => 15000000,
                'description' => '5-floor commercial building with basement parking. Total area: 25000 sq ft.',
                'status' => 'in_progress',
                'client_id' => $clients[4]->id,
            ],
            [
                'project_name' => 'Independent House - Green Meadows',
                'start_date' => Carbon::now()->subMonths(2)->format('Y-m-d'),
                'end_date' => Carbon::now()->addMonths(10)->format('Y-m-d'),
                'budget' => 5500000,
                'description' => '3 BHK independent house with car parking and terrace garden. Area: 2200 sq ft.',
                'status' => 'in_progress',
                'client_id' => $clients[2]->id,
            ],
            [
                'project_name' => 'Duplex Villa - Skyline Heights',
                'start_date' => Carbon::now()->subMonth()->format('Y-m-d'),
                'end_date' => Carbon::now()->addMonths(11)->format('Y-m-d'),
                'budget' => 9800000,
                'description' => 'Premium duplex villa with 5 BHK, home theater, gym, and rooftop terrace. Area: 4000 sq ft.',
                'status' => 'in_progress',
                'client_id' => $clients[3]->id,
            ],
        ];

        foreach ($projectData as $data) {
            $data['created_by'] = $companyUser->creatorId();
            $projects[] = Project::create($data);
        }

        return $projects;
    }

    private function createStagesAndActivities($companyUser, $project)
    {
        // Calculate dates based on project timeline
        $projectStart = Carbon::parse($project->start_date);
        $currentDate = Carbon::now();

        // Define realistic construction stages with activities
        $stagesData = [
            [
                'name' => 'Site Preparation & Foundation',
                'description' => 'Land clearing, excavation, and foundation work',
                'activities' => [
                    ['name' => 'Site Survey and Marking', 'estimated_cost' => 25000, 'actual_cost' => 24500, 'status' => 'completed', 'duration' => 3],
                    ['name' => 'Land Leveling and Excavation', 'estimated_cost' => 85000, 'actual_cost' => 89000, 'status' => 'completed', 'duration' => 5],
                    ['name' => 'Foundation Trenching', 'estimated_cost' => 45000, 'actual_cost' => 47000, 'status' => 'completed', 'duration' => 4],
                    ['name' => 'PCC (Plain Cement Concrete) Work', 'estimated_cost' => 120000, 'actual_cost' => 118000, 'status' => 'completed', 'duration' => 7],
                    ['name' => 'Foundation Column Casting', 'estimated_cost' => 185000, 'actual_cost' => null, 'status' => 'in_progress', 'duration' => 10],
                ],
                'status' => 'in_progress',
                'budget_allocated' => 500000,
            ],
            [
                'name' => 'Structural Work',
                'description' => 'RCC columns, beams, and slab construction',
                'activities' => [
                    ['name' => 'Column Reinforcement', 'estimated_cost' => 220000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 8],
                    ['name' => 'Beam Shuttering', 'estimated_cost' => 150000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 6],
                    ['name' => 'Slab Casting - Ground Floor', 'estimated_cost' => 380000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 12],
                    ['name' => 'First Floor Column Work', 'estimated_cost' => 240000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 10],
                    ['name' => 'Roof Slab Casting', 'estimated_cost' => 420000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 14],
                ],
                'status' => 'pending',
                'budget_allocated' => 1500000,
            ],
            [
                'name' => 'Masonry Work',
                'description' => 'Brick work and block work for walls',
                'activities' => [
                    ['name' => 'External Wall Construction', 'estimated_cost' => 280000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 15],
                    ['name' => 'Internal Partition Walls', 'estimated_cost' => 195000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 10],
                    ['name' => 'Parapet Wall Construction', 'estimated_cost' => 65000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 5],
                ],
                'status' => 'pending',
                'budget_allocated' => 600000,
            ],
            [
                'name' => 'Plumbing & Sanitation',
                'description' => 'Water supply and drainage system installation',
                'activities' => [
                    ['name' => 'Underground Drainage Lines', 'estimated_cost' => 95000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 7],
                    ['name' => 'Water Supply Piping', 'estimated_cost' => 125000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 8],
                    ['name' => 'Bathroom Fixtures Installation', 'estimated_cost' => 185000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 6],
                    ['name' => 'Septic Tank Construction', 'estimated_cost' => 75000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 5],
                ],
                'status' => 'pending',
                'budget_allocated' => 500000,
            ],
            [
                'name' => 'Electrical Work',
                'description' => 'Electrical wiring and fittings',
                'activities' => [
                    ['name' => 'Conduit Installation', 'estimated_cost' => 85000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 6],
                    ['name' => 'Electrical Wiring', 'estimated_cost' => 165000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 10],
                    ['name' => 'Switch Board Installation', 'estimated_cost' => 45000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 3],
                    ['name' => 'Light Fixtures & Fans', 'estimated_cost' => 95000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 4],
                ],
                'status' => 'pending',
                'budget_allocated' => 450000,
            ],
            [
                'name' => 'Plastering & Finishing',
                'description' => 'Internal and external plastering work',
                'activities' => [
                    ['name' => 'External Wall Plastering', 'estimated_cost' => 185000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 12],
                    ['name' => 'Internal Wall Plastering', 'estimated_cost' => 225000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 14],
                    ['name' => 'Ceiling Plastering', 'estimated_cost' => 145000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 8],
                    ['name' => 'Wall Putty Application', 'estimated_cost' => 95000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 7],
                ],
                'status' => 'pending',
                'budget_allocated' => 700000,
            ],
            [
                'name' => 'Flooring & Tiling',
                'description' => 'Floor tiles, bathroom tiles, and marble work',
                'activities' => [
                    ['name' => 'Living Room Marble Flooring', 'estimated_cost' => 285000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 10],
                    ['name' => 'Bedroom Vitrified Tiles', 'estimated_cost' => 195000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 8],
                    ['name' => 'Kitchen Platform Granite', 'estimated_cost' => 125000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 5],
                    ['name' => 'Bathroom Wall & Floor Tiles', 'estimated_cost' => 165000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 7],
                ],
                'status' => 'pending',
                'budget_allocated' => 850000,
            ],
            [
                'name' => 'Doors & Windows',
                'description' => 'Installation of doors, windows, and grills',
                'activities' => [
                    ['name' => 'Main Door Installation', 'estimated_cost' => 85000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 3],
                    ['name' => 'Internal Doors - Flush Doors', 'estimated_cost' => 145000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 6],
                    ['name' => 'UPVC Windows Installation', 'estimated_cost' => 225000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 8],
                    ['name' => 'Window Grills & Safety Bars', 'estimated_cost' => 95000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 5],
                ],
                'status' => 'pending',
                'budget_allocated' => 600000,
            ],
            [
                'name' => 'Painting & Texture',
                'description' => 'Interior and exterior painting work',
                'activities' => [
                    ['name' => 'External Weather Proof Paint', 'estimated_cost' => 125000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 8],
                    ['name' => 'Internal Wall Emulsion Paint', 'estimated_cost' => 185000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 10],
                    ['name' => 'Texture Work - Feature Walls', 'estimated_cost' => 65000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 4],
                    ['name' => 'Wood Work Polishing', 'estimated_cost' => 55000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 5],
                ],
                'status' => 'pending',
                'budget_allocated' => 450000,
            ],
            [
                'name' => 'Final Finishing & Handover',
                'description' => 'Final touches, cleaning, and project completion',
                'activities' => [
                    ['name' => 'Kitchen Modular Fittings', 'estimated_cost' => 285000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 7],
                    ['name' => 'Bathroom Accessories', 'estimated_cost' => 85000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 4],
                    ['name' => 'Final Cleaning & Touch-up', 'estimated_cost' => 35000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 3],
                    ['name' => 'Landscaping & Compound Wall', 'estimated_cost' => 165000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 10],
                    ['name' => 'Final Inspection & Documentation', 'estimated_cost' => 25000, 'actual_cost' => null, 'status' => 'pending', 'duration' => 2],
                ],
                'status' => 'pending',
                'budget_allocated' => 650000,
            ],
        ];

        $order = 0;
        $currentActivityDate = clone $projectStart;

        foreach ($stagesData as $stageData) {
            // Create stage
            $activities = $stageData['activities'];
            unset($stageData['activities']);

            $stage = ConstructionStage::create([
                'project_id' => $project->id,
                'name' => $stageData['name'],
                'description' => $stageData['description'],
                'status' => $stageData['status'],
                'budget_allocated' => $stageData['budget_allocated'],
                'sequence_order' => $order++,
                'created_by' => $companyUser->id,
            ]);

            // Create activities for this stage
            foreach ($activities as $activityData) {
                $startDate = clone $currentActivityDate;
                $endDate = clone $currentActivityDate;
                $endDate->addDays($activityData['duration']);

                ConstructionActivity::create([
                    'construction_stage_id' => $stage->id,
                    'project_id' => $project->id,
                    'name' => $activityData['name'],
                    'estimated_cost' => $activityData['estimated_cost'],
                    'actual_cost' => $activityData['actual_cost'],
                    'status' => $activityData['status'],
                    'progress_percentage' => $activityData['status'] === 'completed' ? 100 : ($activityData['status'] === 'in_progress' ? 50 : 0),
                    'start_date' => $startDate->format('Y-m-d'),
                    'end_date' => $endDate->format('Y-m-d'),
                    'created_by' => $companyUser->id,
                ]);

                // Move to next activity start date (1 day overlap for parallel work)
                $currentActivityDate->addDays($activityData['duration'] - 1);
            }
        }
    }
}
