<?php

namespace App\Services;

use App\Models\Budget;
use App\Models\BudgetAlert;
use App\Models\Bill;
use App\Models\Expense;
use App\Models\User;
use App\Models\Utility;
use Illuminate\Support\Facades\Log;

class BudgetAlertService
{
    // Default thresholds for different alert types
    const WARNING_THRESHOLD = 75.0;   // 75% of budget
    const CRITICAL_THRESHOLD = 90.0;  // 90% of budget
    const OVERRUN_THRESHOLD = 100.0;  // 100% of budget (overrun)

    /**
     * Check all budgets for threshold alerts
     */
    public function checkAllBudgets()
    {
        $budgets = Budget::all();
        $alertsCreated = 0;

        foreach ($budgets as $budget) {
            if ($this->checkBudgetThresholds($budget)) {
                $alertsCreated++;
            }
        }

        return $alertsCreated;
    }

    /**
     * Check a specific budget for threshold alerts
     */
    public function checkBudgetThresholds(Budget $budget)
    {
        $budgetAmount = $this->calculateBudgetAmount($budget);
        $spentAmount = $this->calculateSpentAmount($budget);
        
        if ($budgetAmount <= 0) {
            return false; // Skip if no budget amount
        }

        $currentPercentage = ($spentAmount / $budgetAmount) * 100;
        $alertCreated = false;

        // Check for overrun (highest priority)
        if ($currentPercentage >= self::OVERRUN_THRESHOLD) {
            $this->createOrUpdateAlert($budget, 'overrun', self::OVERRUN_THRESHOLD, $currentPercentage, $budgetAmount, $spentAmount);
            $alertCreated = true;
        }
        // Check for critical threshold
        elseif ($currentPercentage >= self::CRITICAL_THRESHOLD) {
            $this->createOrUpdateAlert($budget, 'critical', self::CRITICAL_THRESHOLD, $currentPercentage, $budgetAmount, $spentAmount);
            $alertCreated = true;
        }
        // Check for warning threshold
        elseif ($currentPercentage >= self::WARNING_THRESHOLD) {
            $this->createOrUpdateAlert($budget, 'warning', self::WARNING_THRESHOLD, $currentPercentage, $budgetAmount, $spentAmount);
            $alertCreated = true;
        }

        return $alertCreated;
    }

    /**
     * Calculate total budget amount from income data
     */
    private function calculateBudgetAmount(Budget $budget)
    {
        $incomeData = json_decode($budget->income_data, true);
        $total = 0;

        if (is_array($incomeData)) {
            foreach ($incomeData as $income) {
                if (isset($income['amount'])) {
                    $total += (float) $income['amount'];
                }
            }
        }

        return $total;
    }

    /**
     * Calculate spent amount from expenses and bills within budget period
     */
    private function calculateSpentAmount(Budget $budget)
    {
        $startDate = $budget->from;
        $endDate = $budget->to;
        $createdBy = $budget->created_by;

        // Calculate expenses
        $expenseTotal = Expense::where('created_by', $createdBy)
            ->whereBetween('date', [$startDate, $endDate])
            ->sum('amount');

        // Calculate bills
        $billTotal = Bill::where('created_by', $createdBy)
            ->whereBetween('bill_date', [$startDate, $endDate])
            ->sum('total');

        return $expenseTotal + $billTotal;
    }

    /**
     * Create or update a budget alert and send notifications
     */
    private function createOrUpdateAlert(Budget $budget, $alertType, $threshold, $currentPercentage, $budgetAmount, $spentAmount)
    {
        // Check if alert already exists for this threshold
        $existingAlert = BudgetAlert::where('budget_id', $budget->id)
            ->where('alert_type', $alertType)
            ->where('status', 'active')
            ->first();

        $isNewAlert = false;
        
        if ($existingAlert) {
            // Update existing alert with current data
            $existingAlert->update([
                'current_percentage' => $currentPercentage,
                'budget_amount' => $budgetAmount,
                'spent_amount' => $spentAmount,
                'message' => $this->generateAlertMessage($alertType, $currentPercentage, $budgetAmount, $spentAmount)
            ]);
            $alert = $existingAlert;
        } else {
            // Create new alert
            $alert = BudgetAlert::create([
                'budget_id' => $budget->id,
                'threshold_percentage' => $threshold,
                'current_percentage' => $currentPercentage,
                'budget_amount' => $budgetAmount,
                'spent_amount' => $spentAmount,
                'alert_type' => $alertType,
                'status' => 'active',
                'message' => $this->generateAlertMessage($alertType, $currentPercentage, $budgetAmount, $spentAmount),
                'triggered_at' => now(),
                'created_by' => $budget->created_by ?? 0
            ]);
            
            $isNewAlert = true;
            Log::info("Budget alert created: {$alertType} for budget '{$budget->name}' at {$currentPercentage}%");
        }

        // Send notifications for new alerts or significant changes
        if ($isNewAlert) {
            $this->sendBudgetAlertNotifications($budget, $alert);
        }
    }

    /**
     * Send budget alert notifications via Slack, Telegram, and Email
     */
    private function sendBudgetAlertNotifications(Budget $budget, BudgetAlert $alert)
    {
        try {
            // Get the budget creator/owner
            $user = User::find($budget->created_by);
            if (!$user) {
                Log::warning("User not found for budget alert notification: budget_id={$budget->id}");
                return;
            }

            // Get notification settings
            $settings = Utility::settingsById($user->id);
            
            // Prepare notification data
            $notificationData = [
                'budget_name' => $budget->name,
                'current_percentage' => number_format($alert->current_percentage, 1) . '%',
                'threshold_percentage' => number_format($alert->threshold_percentage, 1) . '%',
                'spent_amount' => '$' . number_format($alert->spent_amount, 2),
                'budget_amount' => '$' . number_format($alert->budget_amount, 2),
                'alert_type' => ucfirst($alert->alert_type),
                'user_name' => $user->name
            ];

            // Determine notification template slug based on alert type
            $templateSlug = 'budget_' . $alert->alert_type . '_alert';

            // Send Slack notification
            if (isset($settings['budget_notification']) && $settings['budget_notification'] == 1) {
                Utility::send_slack_msg($templateSlug, $notificationData, $user->id);
                Log::info("Slack notification sent for budget alert: {$alert->id}");
            }

            // Send Telegram notification
            if (isset($settings['telegram_budget_notification']) && $settings['telegram_budget_notification'] == 1) {
                Utility::send_telegram_msg($templateSlug, $notificationData, $user->id);
                Log::info("Telegram notification sent for budget alert: {$alert->id}");
            }

            // Send Email notification (if email templates exist)
            if (isset($settings['budget_alert_email']) && $settings['budget_alert_email'] == 1) {
                $emailData = [
                    'budget_name' => $budget->name,
                    'current_percentage' => number_format($alert->current_percentage, 1),
                    'spent_amount' => number_format($alert->spent_amount, 2),
                    'budget_amount' => number_format($alert->budget_amount, 2),
                    'alert_type' => ucfirst($alert->alert_type),
                    'alert_message' => $alert->message
                ];
                
                // Note: Email template would need to be created separately
                // Utility::sendEmailTemplate($templateSlug, [$user->id => $user->email], $emailData);
            }

        } catch (\Exception $e) {
            Log::error("Failed to send budget alert notifications: " . $e->getMessage(), [
                'budget_id' => $budget->id,
                'alert_id' => $alert->id,
                'error' => $e->getTraceAsString()
            ]);
        }
    }

    /**
     * Generate alert message based on type and data
     */
    private function generateAlertMessage($alertType, $currentPercentage, $budgetAmount, $spentAmount)
    {
        $percentage = number_format($currentPercentage, 1);
        $spent = number_format($spentAmount, 2);
        $total = number_format($budgetAmount, 2);

        switch ($alertType) {
            case 'warning':
                return "Budget Warning: {$percentage}% of budget utilized. Spent: \${$spent} of \${$total}";
            case 'critical':
                return "Budget Critical: {$percentage}% of budget utilized. Spent: \${$spent} of \${$total}";
            case 'overrun':
                return "Budget Overrun: {$percentage}% of budget exceeded! Spent: \${$spent} of \${$total}";
            default:
                return "Budget Alert: {$percentage}% of budget utilized. Spent: \${$spent} of \${$total}";
        }
    }

    /**
     * Get active alerts for a specific budget
     */
    public function getActiveBudgetAlerts($budgetId)
    {
        return BudgetAlert::where('budget_id', $budgetId)
            ->where('status', 'active')
            ->orderBy('triggered_at', 'desc')
            ->get();
    }

    /**
     * Get all active alerts across all budgets
     */
    public function getAllActiveAlerts()
    {
        return BudgetAlert::with(['budget', 'acknowledgedBy'])
            ->where('status', 'active')
            ->orderBy('triggered_at', 'desc')
            ->get();
    }

    /**
     * Acknowledge an alert
     */
    public function acknowledgeAlert($alertId, $userId, $notes = null)
    {
        $alert = BudgetAlert::findOrFail($alertId);
        $alert->acknowledge($userId, $notes);
        
        Log::info("Budget alert {$alertId} acknowledged by user {$userId}");
        
        return $alert;
    }

    /**
     * Resolve an alert
     */
    public function resolveAlert($alertId, $userId, $notes = null)
    {
        $alert = BudgetAlert::findOrFail($alertId);
        $alert->resolve($userId, $notes);
        
        Log::info("Budget alert {$alertId} resolved by user {$userId}");
        
        return $alert;
    }
}