<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class PettyCashTransaction extends Model
{
    protected $fillable = [
        'allocation_id',
        'employee_id',
        'expense_id',
        'transaction_type',
        'amount',
        'balance_after',
        'description',
        'receipt_number',
        'transaction_date',
        'project_id',
        'created_by',
    ];

    protected $dates = [
        'transaction_date',
    ];

    protected $casts = [
        'amount' => 'decimal:2',
        'balance_after' => 'decimal:2',
        'transaction_date' => 'datetime',
    ];

    // Transaction type constants
    const TYPE_ALLOCATION = 'allocation';
    const TYPE_EXPENSE = 'expense';
    const TYPE_RETURN = 'return';
    const TYPE_ADJUSTMENT = 'adjustment';

    /**
     * Get the petty cash allocation
     */
    public function allocation()
    {
        return $this->belongsTo(PettyCashAllocation::class, 'allocation_id');
    }

    /**
     * Get the employee who made the transaction
     */
    public function employee()
    {
        return $this->belongsTo(User::class, 'employee_id');
    }

    /**
     * Get the related expense (if any)
     */
    public function expense()
    {
        return $this->belongsTo(Expense::class, 'expense_id');
    }

    /**
     * Get the related project (if any)
     */
    public function project()
    {
        return $this->belongsTo(Project::class, 'project_id');
    }

    /**
     * Get the user who created the transaction
     */
    public function createdBy()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Scope for specific transaction type
     */
    public function scopeOfType($query, $type)
    {
        return $query->where('transaction_type', $type);
    }

    /**
     * Scope for specific employee
     */
    public function scopeForEmployee($query, $employeeId)
    {
        return $query->where('employee_id', $employeeId);
    }

    /**
     * Scope for specific allocation
     */
    public function scopeForAllocation($query, $allocationId)
    {
        return $query->where('allocation_id', $allocationId);
    }

    /**
     * Scope for date range
     */
    public function scopeDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('transaction_date', [$startDate, $endDate]);
    }

    /**
     * Get formatted amount
     */
    public function getFormattedAmountAttribute()
    {
        return number_format($this->amount, 2);
    }

    /**
     * Get formatted balance after
     */
    public function getFormattedBalanceAfterAttribute()
    {
        return number_format($this->balance_after, 2);
    }

    /**
     * Get transaction type label
     */
    public function getTransactionTypeLabelAttribute()
    {
        $labels = [
            self::TYPE_ALLOCATION => 'Allocation',
            self::TYPE_EXPENSE => 'Expense',
            self::TYPE_RETURN => 'Return',
            self::TYPE_ADJUSTMENT => 'Adjustment',
        ];

        return $labels[$this->transaction_type] ?? 'Unknown';
    }

    /**
     * Create a new transaction record
     */
    public static function createTransaction($data)
    {
        return self::create([
            'allocation_id' => $data['allocation_id'],
            'employee_id' => $data['employee_id'],
            'expense_id' => $data['expense_id'] ?? null,
            'transaction_type' => $data['transaction_type'],
            'amount' => $data['amount'],
            'balance_after' => $data['balance_after'],
            'description' => $data['description'] ?? null,
            'receipt_number' => $data['receipt_number'] ?? null,
            'transaction_date' => $data['transaction_date'] ?? now(),
            'project_id' => $data['project_id'] ?? null,
            'created_by' => auth()->id(),
        ]);
    }
}