<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class PettyCashAllocation extends Model
{
    use SoftDeletes;

    protected $fillable = [
        'employee_id',
        'allocated_amount',
        'current_balance',
        'allocation_date',
        'allocated_by',
        'purpose',
        'status',
    ];

    protected $dates = [
        'allocation_date',
        'deleted_at',
    ];

    protected $casts = [
        'allocated_amount' => 'decimal:2',
        'current_balance' => 'decimal:2',
        'allocation_date' => 'date',
    ];

    // Status constants
    const STATUS_ACTIVE = 'active';
    const STATUS_COMPLETED = 'completed';
    const STATUS_CANCELLED = 'cancelled';

    /**
     * Get the employee who received the allocation
     */
    public function employee()
    {
        return $this->belongsTo(User::class, 'employee_id');
    }

    /**
     * Get the user who allocated the petty cash
     */
    public function allocatedBy()
    {
        return $this->belongsTo(User::class, 'allocated_by');
    }

    /**
     * Get all transactions for this allocation
     */
    public function transactions()
    {
        return $this->hasMany(PettyCashTransaction::class, 'allocation_id');
    }

    /**
     * Get all expenses related to this allocation
     */
    public function expenses()
    {
        return $this->hasMany(Expense::class, 'petty_cash_allocation_id');
    }

    /**
     * Calculate the spent amount
     */
    public function getSpentAmountAttribute()
    {
        return $this->transactions()
            ->where('transaction_type', 'expense')
            ->sum('amount');
    }

    /**
     * Calculate the remaining balance
     */
    public function getRemainingBalanceAttribute()
    {
        return $this->allocated_amount - $this->spent_amount;
    }

    /**
     * Check if allocation has sufficient balance
     */
    public function hasSufficientBalance($amount)
    {
        return $this->current_balance >= $amount;
    }

    /**
     * Update the current balance
     */
    public function updateBalance($amount, $type = 'expense')
    {
        if ($type === 'expense') {
            $this->current_balance -= $amount;
        } elseif ($type === 'return') {
            $this->current_balance += $amount;
        }

        // Update status if balance is zero or negative
        if ($this->current_balance <= 0) {
            $this->status = self::STATUS_COMPLETED;
        }

        $this->save();
    }

    /**
     * Scope for active allocations
     */
    public function scopeActive($query)
    {
        return $query->where('status', self::STATUS_ACTIVE);
    }

    /**
     * Scope for specific employee
     */
    public function scopeForEmployee($query, $employeeId)
    {
        return $query->where('employee_id', $employeeId);
    }

    /**
     * Get formatted allocation amount
     */
    public function getFormattedAllocatedAmountAttribute()
    {
        return number_format($this->allocated_amount, 2);
    }

    /**
     * Get formatted current balance
     */
    public function getFormattedCurrentBalanceAttribute()
    {
        return number_format($this->current_balance, 2);
    }
}