<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class ConstructionStage extends Model
{
    protected $fillable = [
        'project_id',
        'construction_floor_id',
        'name',
        'description',
        'sequence_order',
        'budget_allocated',
        'start_date',
        'end_date',
        'status',
        'created_by',
    ];

    protected $casts = [
        'budget_allocated' => 'decimal:2',
        'start_date' => 'date',
        'end_date' => 'date',
    ];

    public static $statuses = [
        'pending' => 'Pending',
        'in_progress' => 'In Progress',
        'completed' => 'Completed',
        'on_hold' => 'On Hold',
    ];

    public static $status_colors = [
        'pending' => 'secondary',
        'in_progress' => 'primary',
        'completed' => 'success',
        'on_hold' => 'warning',
    ];

    // Relationships
    public function project()
    {
        return $this->belongsTo(Project::class);
    }

    public function floor()
    {
        return $this->belongsTo(ConstructionFloor::class, 'construction_floor_id');
    }

    public function activities()
    {
        return $this->hasMany(ConstructionActivity::class)->orderBy('id', 'desc');
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    // Helper Methods
    public function getBudgetSpent()
    {
        return $this->activities()->sum('actual_cost') ?? 0;
    }

    public function getBudgetRemaining()
    {
        if (!$this->budget_allocated)
            return null;
        return $this->budget_allocated - $this->getBudgetSpent();
    }

    public function getProgressPercentage()
    {
        $total = $this->activities()->count();
        if ($total == 0)
            return 0;

        $completed = $this->activities()->where('status', 'completed')->count();
        return round(($completed / $total) * 100);
    }

    public function getBudgetUtilizationPercentage()
    {
        if (!$this->budget_allocated || $this->budget_allocated == 0)
            return 0;
        return round(($this->getBudgetSpent() / $this->budget_allocated) * 100);
    }

    public function isOverBudget()
    {
        if (!$this->budget_allocated)
            return false;
        return $this->getBudgetSpent() > $this->budget_allocated;
    }

    public function hasBudget()
    {
        return !is_null($this->budget_allocated) && $this->budget_allocated > 0;
    }
}
