<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class ConstructionActivity extends Model
{
    protected $fillable = [
        'construction_stage_id',
        'project_id',
        'name',
        'description',
        'estimated_cost',
        'actual_cost',
        'labor_hours',
        'material_cost',
        'equipment_cost',
        'status',
        'progress_percentage',
        'assigned_to',
        'start_date',
        'end_date',
        'created_by',
    ];

    protected $casts = [
        'estimated_cost' => 'decimal:2',
        'actual_cost' => 'decimal:2',
        'labor_hours' => 'decimal:2',
        'material_cost' => 'decimal:2',
        'equipment_cost' => 'decimal:2',
        'start_date' => 'date',
        'end_date' => 'date',
    ];

    public static $statuses = [
        'pending' => 'Pending',
        'in_progress' => 'In Progress',
        'completed' => 'Completed',
        'rework' => 'Rework Required',
    ];

    public static $status_colors = [
        'pending' => 'secondary',
        'in_progress' => 'primary',
        'completed' => 'success',
        'rework' => 'danger',
    ];

    // Relationships
    public function stage()
    {
        return $this->belongsTo(ConstructionStage::class, 'construction_stage_id');
    }

    public function project()
    {
        return $this->belongsTo(Project::class);
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function assignedUsers()
    {
        if (empty($this->assigned_to)) {
            return collect([]);
        }
        return User::whereIn('id', explode(',', $this->assigned_to))->get();
    }

    // Helper Methods
    public function getCostVariance()
    {
        if (!$this->estimated_cost) return null;
        return $this->estimated_cost - ($this->actual_cost ?? 0);
    }

    public function isOverBudget()
    {
        if (!$this->estimated_cost) return false;
        return ($this->actual_cost ?? 0) > $this->estimated_cost;
    }

    public function getCostUtilizationPercentage()
    {
        if (!$this->estimated_cost || $this->estimated_cost == 0) return 0;
        return round((($this->actual_cost ?? 0) / $this->estimated_cost) * 100);
    }

    public function getStatusBadgeColor()
    {
        return self::$status_colors[$this->status] ?? 'secondary';
    }

    public function getStatusLabel()
    {
        return self::$statuses[$this->status] ?? 'Unknown';
    }

    public function hasEstimate()
    {
        return !is_null($this->estimated_cost) && $this->estimated_cost > 0;
    }

    public function getTotalCost()
    {
        return ($this->labor_cost ?? 0) + ($this->material_cost ?? 0) + ($this->equipment_cost ?? 0);
    }
}
