<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class BudgetAlert extends Model
{
    use HasFactory;

    protected $fillable = [
        'budget_id',
        'alert_type',
        'threshold_percentage',
        'current_percentage',
        'budget_amount',
        'spent_amount',
        'message',
        'status',
        'triggered_at',
        'acknowledged_at',
        'acknowledged_by',
        'resolved_at',
        'resolved_by',
        'resolution_notes',
        'notes',
        'created_by'
    ];

    protected $casts = [
        'threshold_percentage' => 'decimal:2',
        'current_percentage' => 'decimal:2',
        'budget_amount' => 'decimal:2',
        'spent_amount' => 'decimal:2',
        'triggered_at' => 'datetime',
        'acknowledged_at' => 'datetime',
    ];

    /**
     * Get the budget that owns the alert.
     */
    public function budget(): BelongsTo
    {
        return $this->belongsTo(Budget::class);
    }

    /**
     * Get the user who acknowledged the alert.
     */
    public function acknowledgedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'acknowledged_by');
    }

    /**
     * Get the user who created the alert.
     */
    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Scope to get active alerts.
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    /**
     * Scope to get alerts by type.
     */
    public function scopeByType($query, $type)
    {
        return $query->where('alert_type', $type);
    }

    /**
     * Check if alert is acknowledged.
     */
    public function isAcknowledged(): bool
    {
        return $this->status === 'acknowledged' || $this->status === 'resolved';
    }

    /**
     * Acknowledge the alert.
     */
    public function acknowledge($userId, $notes = null)
    {
        $this->update([
            'status' => 'acknowledged',
            'acknowledged_at' => now(),
            'acknowledged_by' => $userId,
            'resolution_notes' => $notes
        ]);
    }

    /**
     * Resolve the alert.
     */
    public function resolve($userId, $notes = null)
    {
        $this->update([
            'status' => 'resolved',
            'acknowledged_at' => now(),
            'acknowledged_by' => $userId,
            'resolution_notes' => $notes
        ]);
    }
}
