<?php

namespace App\Http\Controllers;

use App\Models\PettyCashAllocation;
use App\Models\PettyCashTransaction;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class PettyCashController extends Controller
{
    /**
     * Display a listing of petty cash allocations.
     */
    public function index()
    {
        $allocations = PettyCashAllocation::with(['employee', 'allocatedBy'])
            ->orderBy('created_at', 'desc')
            ->paginate(15);

        return view('petty_cash.index', compact('allocations'));
    }

    /**
     * Show the form for creating a new allocation.
     */
    public function create()
    {
        $employees = User::where('type', 'employee')->get();
        return view('petty_cash.create', compact('employees'));
    }

    /**
     * Store a newly created allocation.
     */
    public function store(Request $request)
    {
        $request->validate([
            'employee_id' => 'required|exists:users,id',
            'allocated_amount' => 'required|numeric|min:1',
            'allocation_date' => 'required|date',
            'purpose' => 'required|string|max:255',
        ]);

        DB::beginTransaction();
        try {
            // Create allocation
            $allocation = PettyCashAllocation::create([
                'employee_id' => $request->employee_id,
                'allocated_amount' => $request->allocated_amount,
                'current_balance' => $request->allocated_amount,
                'allocation_date' => $request->allocation_date,
                'allocated_by' => Auth::id(),
                'purpose' => $request->purpose,
                'status' => PettyCashAllocation::STATUS_ACTIVE,
            ]);

            // Create initial transaction record
            PettyCashTransaction::createTransaction([
                'allocation_id' => $allocation->id,
                'employee_id' => $request->employee_id,
                'transaction_type' => PettyCashTransaction::TYPE_ALLOCATION,
                'amount' => $request->allocated_amount,
                'balance_after' => $request->allocated_amount,
                'description' => 'Initial petty cash allocation: ' . $request->purpose,
                'transaction_date' => $request->allocation_date,
            ]);

            DB::commit();

            return redirect()->route('petty-cash.index')
                ->with('success', 'Petty cash allocated successfully.');

        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->back()
                ->with('error', 'Error allocating petty cash: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Display the specified allocation.
     */
    public function show($id)
    {
        $allocation = PettyCashAllocation::with(['employee', 'allocatedBy', 'transactions.expense'])
            ->findOrFail($id);

        $transactions = $allocation->transactions()
            ->with(['expense', 'project'])
            ->orderBy('transaction_date', 'desc')
            ->get();

        return view('petty_cash.show', compact('allocation', 'transactions'));
    }

    /**
     * Get active allocations for a specific employee (AJAX)
     */
    public function getEmployeeAllocations(Request $request)
    {
        $employeeId = $request->get('employee_id');
        
        $allocations = PettyCashAllocation::active()
            ->forEmployee($employeeId)
            ->where('current_balance', '>', 0)
            ->get(['id', 'purpose', 'current_balance', 'allocated_amount']);

        return response()->json($allocations);
    }

    /**
     * Check allocation balance (AJAX)
     */
    public function checkBalance(Request $request)
    {
        $allocationId = $request->get('allocation_id');
        
        $allocation = PettyCashAllocation::find($allocationId);
        
        if (!$allocation) {
            return response()->json(['error' => 'Allocation not found'], 404);
        }

        return response()->json([
            'current_balance' => $allocation->current_balance,
            'formatted_balance' => $allocation->formatted_current_balance,
        ]);
    }

    /**
     * Return unused petty cash
     */
    public function returnCash(Request $request, $id)
    {
        $request->validate([
            'return_amount' => 'required|numeric|min:0.01',
            'description' => 'nullable|string|max:255',
        ]);

        $allocation = PettyCashAllocation::findOrFail($id);
        
        if ($request->return_amount > $allocation->current_balance) {
            return redirect()->back()
                ->with('error', 'Return amount cannot exceed current balance.');
        }

        DB::beginTransaction();
        try {
            // Update allocation balance
            $allocation->updateBalance($request->return_amount, 'return');

            // Create return transaction
            PettyCashTransaction::createTransaction([
                'allocation_id' => $allocation->id,
                'employee_id' => $allocation->employee_id,
                'transaction_type' => PettyCashTransaction::TYPE_RETURN,
                'amount' => $request->return_amount,
                'balance_after' => $allocation->current_balance,
                'description' => $request->description ?? 'Petty cash return',
                'transaction_date' => now(),
            ]);

            DB::commit();

            return redirect()->back()
                ->with('success', 'Petty cash returned successfully.');

        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->back()
                ->with('error', 'Error returning petty cash: ' . $e->getMessage());
        }
    }

    /**
     * Get petty cash summary for dashboard
     */
    public function getSummary()
    {
        $summary = [
            'total_allocated' => PettyCashAllocation::sum('allocated_amount'),
            'total_balance' => PettyCashAllocation::active()->sum('current_balance'),
            'total_spent' => PettyCashTransaction::ofType('expense')->sum('amount'),
            'active_allocations' => PettyCashAllocation::active()->count(),
        ];

        return response()->json($summary);
    }
}