<?php

namespace App\Http\Controllers;

use App\Models\Lead;
use App\Models\LeadQuotation;
use App\Models\LeadQuotationItem;
use App\Models\Brand;
use App\Models\TaskMaster;
use Illuminate\Http\Request;

class CostCalculatorController extends Controller
{
    public function getQuotationData($leadId)
    {
        $lead = Lead::findOrFail($leadId);
        $quotation = LeadQuotation::firstOrCreate(
            ['lead_id' => $leadId],
            [
                'created_by' => \Auth::user()->creatorId(),
                'status' => 'draft'
            ]
        );

        $items = $quotation->items()->with('brand')->get();
        $brands = Brand::orderBy('name')->get();
        // Assuming we might want standard tasks
        // $tasks = TaskMaster::all(); 

        return response()->json([
            'quotation' => $quotation,
            'items' => $items,
            'brands' => $brands,
        ]);
    }

    public function ensureQuotation($leadId)
    {
        return LeadQuotation::firstOrCreate(
            ['lead_id' => $leadId],
            [
                'created_by' => \Auth::user()->creatorId(),
                'status' => 'draft'
            ]
        );
    }

    public function storeItem(Request $request, $leadId)
    {
        $quotation = $this->ensureQuotation($leadId);

        $validator = \Validator::make($request->all(), [
            'floor_name' => 'required',
            'task_name' => 'required',
            'quantity' => 'required|numeric',
            'rate' => 'required|numeric',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        $amount = $request->quantity * $request->rate;

        $item = LeadQuotationItem::create([
            'quotation_id' => $quotation->id,
            'floor_name' => $request->floor_name,
            'task_name' => $request->task_name,
            'material_spec' => $request->material_spec,
            'brand_id' => $request->brand_id,
            'unit' => $request->unit,
            'quantity' => $request->quantity,
            'rate' => $request->rate,
            'amount' => $amount,
        ]);

        $this->updateTotal($quotation);

        return response()->json(['success' => 'Item added', 'item' => $item]);
    }

    public function updateItem(Request $request, $leadId, $itemId)
    {
        $item = LeadQuotationItem::findOrFail($itemId);

        $amount = $request->quantity * $request->rate;

        $item->update([
            'floor_name' => $request->floor_name,
            'task_name' => $request->task_name,
            'material_spec' => $request->material_spec,
            'brand_id' => $request->brand_id,
            'unit' => $request->unit,
            'quantity' => $request->quantity,
            'rate' => $request->rate,
            'amount' => $amount,
        ]);

        $this->updateTotal($item->quotation);

        return response()->json(['success' => 'Item updated', 'item' => $item]);
    }

    public function deleteItem($leadId, $itemId)
    {
        $item = LeadQuotationItem::findOrFail($itemId);
        $quotation = $item->quotation;
        $item->delete();

        $this->updateTotal($quotation);

        return response()->json(['success' => 'Item deleted']);
    }

    private function updateTotal($quotation)
    {
        $total = $quotation->items()->sum('amount');
        $quotation->update(['total_amount' => $total]);
    }
}
