<?php

namespace App\Http\Controllers;

use App\Models\ConstructionTask;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class ConstructionTaskController extends Controller
{
    public function index()
    {
        if (Auth::user()->can('manage project')) {
            // Show tasks created by current user OR by Admin (ID 1)
            $tasks = ConstructionTask::where('created_by', '=', \Auth::user()->creatorId())
                ->orWhere('created_by', '=', 1)
                ->get();
            return view('construction.tasks.index', compact('tasks'));
        } else {
            return redirect()->back()->with('error', __('Permission Denied.'));
        }
    }

    public function create()
    {
        if (Auth::user()->can('create project')) {
            return view('construction.tasks.create');
        } else {
            return response()->json(['error' => __('Permission Denied.')], 401);
        }
    }

    public function store(Request $request)
    {
        if (Auth::user()->can('create project')) {
            $validator = Validator::make($request->all(), [
                'name' => 'required',
            ]);

            if ($validator->fails()) {
                return redirect()->back()->with('errors', $validator->errors());
            }

            $task = new ConstructionTask();
            $task->name = $request->name;
            $task->code = $request->code;
            $task->is_active = $request->is_active ? 1 : 0;
            $task->description = $request->description;
            $task->created_by = \Auth::user()->creatorId();
            $task->save();

            return redirect()->route('construction-tasks.index')->with('success', __('Task successfully created.'));
        } else {
            return redirect()->back()->with('error', __('Permission Denied.'));
        }
    }

    public function edit(ConstructionTask $constructionTask)
    {
        if (Auth::user()->can('edit project')) {
            return view('construction.tasks.edit', compact('constructionTask'));
        } else {
            return response()->json(['error' => __('Permission Denied.')], 401);
        }
    }

    public function update(Request $request, ConstructionTask $constructionTask)
    {
        if (Auth::user()->can('edit project')) {
            $validator = Validator::make($request->all(), [
                'name' => 'required',
            ]);

            if ($validator->fails()) {
                return redirect()->back()->with('errors', $validator->errors());
            }

            $constructionTask->name = $request->name;
            $constructionTask->code = $request->code;
            $constructionTask->is_active = $request->is_active ? 1 : 0;
            $constructionTask->description = $request->description;
            $constructionTask->save();

            return redirect()->route('construction-tasks.index')->with('success', __('Task successfully updated.'));
        } else {
            return redirect()->back()->with('error', __('Permission Denied.'));
        }
    }

    public function destroy(ConstructionTask $constructionTask)
    {
        if (Auth::user()->can('delete project')) {
            $constructionTask->delete();
            return redirect()->route('construction-tasks.index')->with('success', __('Task successfully deleted.'));
        } else {
            return redirect()->back()->with('error', __('Permission Denied.'));
        }
    }
}
