<?php

namespace App\Http\Controllers;

use App\Models\Project;
use App\Models\ConstructionStage;
use App\Models\ConstructionFloor;
use App\Models\ConstructionTask;
use App\Models\Utility;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class ConstructionStageController extends Controller
{
    public function index($project_id)
    {
        if (Auth::user()->can('manage project')) {
            $project = Project::findOrFail($project_id);

            // Check if user has access to this project
            $usr = Auth::user();
            if ($usr->type == 'client') {
                if ($project->client_id != $usr->id) {
                    return redirect()->back()->with('error', __('Permission Denied.'));
                }
            } elseif ($project->created_by != $usr->creatorId()) {
                return redirect()->back()->with('error', __('Permission Denied.'));
            }

            $stages = $project->constructionStages()->whereNull('construction_floor_id')->orderBy('sequence_order')->get();
            $floors = $project->constructionFloors()->with([
                'stages' => function ($query) {
                    $query->orderBy('sequence_order');
                }
            ])->orderBy('sequence_order')->get();

            return view('construction.stages.index', compact('project', 'stages', 'floors'));
        }
        return redirect()->back()->with('error', __('Permission Denied.'));
    }

    public function create($project_id)
    {
        if (Auth::user()->can('create project')) {
            $project = Project::findOrFail($project_id);
            $floors = \App\Models\ConstructionFloor::where('project_id', $project_id)->orderBy('sequence_order')->pluck('name', 'id');
            return view('construction.stages.create', compact('project', 'floors'));
        }
        return redirect()->back()->with('error', __('Permission Denied.'));
    }

    public function store(Request $request, $project_id)
    {
        if (Auth::user()->can('create project')) {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'budget_allocated' => 'nullable|numeric|min:0',
            ]);

            if ($validator->fails()) {
                return redirect()->back()->with('error', Utility::errorFormat($validator->getMessageBag()));
            }

            $stage = new ConstructionStage();
            $stage->project_id = $project_id;
            $stage->construction_floor_id = $request->construction_floor_id;
            $stage->name = $request->name;
            $stage->description = $request->description;
            $stage->budget_allocated = $request->budget_allocated;
            $stage->start_date = $request->start_date;
            $stage->end_date = $request->end_date;
            $stage->status = $request->status ?? 'pending';
            $stage->sequence_order = ConstructionStage::where('project_id', $project_id)->max('sequence_order') + 1;
            $stage->created_by = Auth::user()->id;
            $stage->save();

            return redirect()->route('construction.stages.index', $project_id)
                ->with('success', __('Construction stage created successfully.'));
        }
        return redirect()->back()->with('error', __('Permission Denied.'));
    }

    public function edit($id)
    {
        if (Auth::user()->can('edit project')) {
            $stage = ConstructionStage::findOrFail($id);
            $floors = \App\Models\ConstructionFloor::where('project_id', $stage->project_id)->orderBy('sequence_order')->pluck('name', 'id');
            return view('construction.stages.edit', compact('stage', 'floors'));
        }
        return redirect()->back()->with('error', __('Permission Denied.'));
    }

    public function update(Request $request, $id)
    {
        if (Auth::user()->can('edit project')) {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'budget_allocated' => 'nullable|numeric|min:0',
            ]);

            if ($validator->fails()) {
                return redirect()->back()->with('error', Utility::errorFormat($validator->getMessageBag()));
            }

            $stage = ConstructionStage::findOrFail($id);
            $stage->construction_floor_id = $request->construction_floor_id;
            $stage->name = $request->name;
            $stage->description = $request->description;
            $stage->budget_allocated = $request->budget_allocated;
            $stage->start_date = $request->start_date;
            $stage->end_date = $request->end_date;
            $stage->status = $request->status;
            $stage->save();

            return redirect()->route('construction.stages.index', $stage->project_id)
                ->with('success', __('Construction stage updated successfully.'));
        }
        return redirect()->back()->with('error', __('Permission Denied.'));
    }

    public function destroy($id)
    {
        if (Auth::user()->can('delete project')) {
            $stage = ConstructionStage::findOrFail($id);
            $project_id = $stage->project_id;
            $stage->delete();

            return redirect()->route('construction.stages.index', $project_id)
                ->with('success', __('Construction stage deleted successfully.'));
        }
        return redirect()->back()->with('error', __('Permission Denied.'));
    }

    public function updateOrder(Request $request, $project_id)
    {
        if (Auth::user()->can('edit project')) {
            $order = $request->order;
            foreach ($order as $index => $id) {
                ConstructionStage::where('id', $id)->update(['sequence_order' => $index]);
            }
            return response()->json(['success' => true]);
        }
        return response()->json(['error' => 'Permission Denied'], 403);
    }

    public function show($id)
    {
        if (Auth::user()->can('view project')) {
            $stage = ConstructionStage::with(['activities', 'project'])->findOrFail($id);

            return view('construction.stages.show', compact('stage'));
        }
        return redirect()->back()->with('error', __('Permission Denied.'));
    }

    public function library($project_id)
    {
        if (Auth::user()->can('create project')) {
            $project = Project::find($project_id);
            $floors = ConstructionFloor::where('project_id', $project_id)->get()->pluck('name', 'id');
            $floors->prepend('Unassigned', '');
            $tasks = \App\Models\ConstructionTask::where('is_active', 1)->get();
            return view('construction.stages.library', compact('project', 'floors', 'tasks'));
        } else {
            return response()->json(['error' => __('Permission Denied.')], 401);
        }
    }

    public function import(Request $request, $project_id)
    {
        if (Auth::user()->can('create project')) {
            $validator = Validator::make($request->all(), [
                'tasks' => 'required|array',
                'tasks.*' => 'exists:construction_tasks,id',
            ]);

            if ($validator->fails()) {
                return redirect()->back()->with('errors', $validator->errors());
            }

            foreach ($request->tasks as $taskId) {
                $task = \App\Models\ConstructionTask::find($taskId);
                $stage = new ConstructionStage();
                $stage->name = $task->name;
                $stage->project_id = $project_id;
                $stage->construction_floor_id = $request->construction_floor_id;
                $stage->budget_allocated = 0; // Default
                $stage->status = 'pending';
                $stage->start_date = date('Y-m-d');
                $stage->end_date = date('Y-m-d');
                $stage->created_by = \Auth::user()->creatorId();
                // Determine sequence order
                $lastStage = ConstructionStage::where('project_id', $project_id)
                    ->where('construction_floor_id', $request->construction_floor_id)
                    ->orderBy('sequence_order', 'desc')
                    ->first();
                $stage->sequence_order = $lastStage ? $lastStage->sequence_order + 1 : 1;
                $stage->save();
            }

            return redirect()->route('construction.stages.index', $project_id)->with('success', __('Stages imported from library.'));
        } else {
            return redirect()->back()->with('error', __('Permission Denied.'));
        }
    }
}
