<?php

namespace App\Http\Controllers;

use App\Models\ConstructionFloor;
use App\Models\Project; // Corrected Namespace
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ConstructionFloorController extends Controller
{
    public function index(Project $project)
    {
        if (Auth::user()->can('manage project')) {
            $floors = $project->constructionFloors;
            return view('projects.construction.floors.index', compact('project', 'floors'));
        }
        return redirect()->back()->with('error', __('Permission denied.'));
    }

    public function create(Project $project)
    {
        if (Auth::user()->can('manage project')) {
            return view('projects.construction.floors.create', compact('project'));
        }
        return response()->json(['error' => __('Permission denied.')], 401);
    }

    public function store(Request $request, Project $project)
    {
        if (Auth::user()->can('manage project')) {
            $validator = \Validator::make($request->all(), [
                'name' => 'required|string|max:255',
            ]);

            if ($validator->fails()) {
                return redirect()->back()->with('error', $validator->errors()->first());
            }

            ConstructionFloor::create([
                'project_id' => $project->id,
                'name' => $request->name,
                'description' => $request->description,
                'sequence_order' => ConstructionFloor::where('project_id', $project->id)->max('sequence_order') + 1,
            ]);

            return redirect()->back()->with('success', __('Floor successfully created.'));
        }
        return redirect()->back()->with('error', __('Permission denied.'));
    }

    public function edit(ConstructionFloor $floor)
    {
        if (Auth::user()->can('manage project')) {
            return view('projects.construction.floors.edit', compact('floor'));
        }
        return response()->json(['error' => __('Permission denied.')], 401);
    }

    public function update(Request $request, ConstructionFloor $floor)
    {
        if (Auth::user()->can('manage project')) {
            $validator = \Validator::make($request->all(), [
                'name' => 'required|string|max:255',
            ]);

            if ($validator->fails()) {
                return redirect()->back()->with('error', $validator->errors()->first());
            }

            $floor->update($request->all());

            return redirect()->back()->with('success', __('Floor successfully updated.'));
        }
        return redirect()->back()->with('error', __('Permission denied.'));
    }

    public function destroy(ConstructionFloor $floor)
    {
        if (Auth::user()->can('manage project')) {
            $floor->delete();
            return redirect()->back()->with('success', __('Floor successfully deleted.'));
        }
        return redirect()->back()->with('error', __('Permission denied.'));
    }

    public function updateOrder(Request $request, Project $project)
    {
        if (Auth::user()->can('manage project')) {
            $order = $request->order; // Array of floor IDs in desired order
            foreach ($order as $index => $floorId) {
                ConstructionFloor::where('id', $floorId)->update(['sequence_order' => $index]);
            }
            return response()->json(['status' => 'success']);
        }
        return response()->json(['status' => 'error'], 403);
    }
}
