<?php

namespace App\Http\Controllers;

use App\Models\Project;
use App\Models\User;
use App\Models\ConstructionStage;
use App\Models\ConstructionActivity;
use App\Models\Utility;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class ConstructionActivityController extends Controller
{
    public function index($stage_id)
    {
        if (Auth::user()->can('manage project')) {
            $stage = ConstructionStage::with(['project', 'activities'])->findOrFail($stage_id);
            $project = $stage->project;

            // Check if user has access to this project
            $usr = Auth::user();
            if ($usr->type == 'client') {
                if ($project->client_id != $usr->id) {
                    return redirect()->back()->with('error', __('Permission Denied.'));
                }
            } elseif ($project->created_by != $usr->creatorId()) {
                return redirect()->back()->with('error', __('Permission Denied.'));
            }

            $activities = $stage->activities;

            return view('construction.activities.index', compact('stage', 'project', 'activities'));
        }
        return redirect()->back()->with('error', __('Permission Denied.'));
    }

    public function create($stage_id)
    {
        if (Auth::user()->can('create project')) {
            $stage = ConstructionStage::with('project')->findOrFail($stage_id);
            $project = $stage->project;

            // Get users assigned to the project
            $users = User::where('created_by', '=', Auth::user()->creatorId())
                         ->where('type', '!=', 'client')
                         ->get()
                         ->pluck('name', 'id');

            return view('construction.activities.create', compact('stage', 'project', 'users'));
        }
        return redirect()->back()->with('error', __('Permission Denied.'));
    }

    public function store(Request $request, $stage_id)
    {
        if (Auth::user()->can('create project')) {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'estimated_cost' => 'nullable|numeric|min:0',
                'actual_cost' => 'nullable|numeric|min:0',
                'labor_hours' => 'nullable|numeric|min:0',
                'material_cost' => 'nullable|numeric|min:0',
                'equipment_cost' => 'nullable|numeric|min:0',
            ]);

            if ($validator->fails()) {
                return redirect()->back()->with('error', Utility::errorFormat($validator->getMessageBag()));
            }

            $stage = ConstructionStage::findOrFail($stage_id);

            $activity = new ConstructionActivity();
            $activity->construction_stage_id = $stage_id;
            $activity->project_id = $stage->project_id;
            $activity->name = $request->name;
            $activity->description = $request->description;
            $activity->estimated_cost = $request->estimated_cost;
            $activity->actual_cost = $request->actual_cost;
            $activity->labor_hours = $request->labor_hours;
            $activity->material_cost = $request->material_cost;
            $activity->equipment_cost = $request->equipment_cost;
            $activity->status = $request->status ?? 'pending';
            $activity->progress_percentage = $request->progress_percentage ?? 0;
            $activity->assigned_to = !empty($request->assigned_to) ? implode(',', $request->assigned_to) : null;
            $activity->start_date = $request->start_date;
            $activity->end_date = $request->end_date;
            $activity->created_by = Auth::user()->id;
            $activity->save();

            return redirect()->route('construction.activities.index', $stage_id)
                           ->with('success', __('Construction activity created successfully.'));
        }
        return redirect()->back()->with('error', __('Permission Denied.'));
    }

    public function edit($id)
    {
        if (Auth::user()->can('edit project')) {
            $activity = ConstructionActivity::with(['stage.project'])->findOrFail($id);

            // Get users assigned to the project
            $users = User::where('created_by', '=', Auth::user()->creatorId())
                         ->where('type', '!=', 'client')
                         ->get()
                         ->pluck('name', 'id');

            return view('construction.activities.edit', compact('activity', 'users'));
        }
        return redirect()->back()->with('error', __('Permission Denied.'));
    }

    public function update(Request $request, $id)
    {
        if (Auth::user()->can('edit project')) {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'estimated_cost' => 'nullable|numeric|min:0',
                'actual_cost' => 'nullable|numeric|min:0',
                'labor_hours' => 'nullable|numeric|min:0',
                'material_cost' => 'nullable|numeric|min:0',
                'equipment_cost' => 'nullable|numeric|min:0',
            ]);

            if ($validator->fails()) {
                return redirect()->back()->with('error', Utility::errorFormat($validator->getMessageBag()));
            }

            $activity = ConstructionActivity::findOrFail($id);
            $activity->name = $request->name;
            $activity->description = $request->description;
            $activity->estimated_cost = $request->estimated_cost;
            $activity->actual_cost = $request->actual_cost;
            $activity->labor_hours = $request->labor_hours;
            $activity->material_cost = $request->material_cost;
            $activity->equipment_cost = $request->equipment_cost;
            $activity->status = $request->status;
            $activity->progress_percentage = $request->progress_percentage ?? 0;
            $activity->assigned_to = !empty($request->assigned_to) ? implode(',', $request->assigned_to) : null;
            $activity->start_date = $request->start_date;
            $activity->end_date = $request->end_date;
            $activity->save();

            return redirect()->route('construction.activities.index', $activity->construction_stage_id)
                           ->with('success', __('Construction activity updated successfully.'));
        }
        return redirect()->back()->with('error', __('Permission Denied.'));
    }

    public function destroy($id)
    {
        if (Auth::user()->can('delete project')) {
            $activity = ConstructionActivity::findOrFail($id);
            $stage_id = $activity->construction_stage_id;
            $activity->delete();

            return redirect()->route('construction.activities.index', $stage_id)
                           ->with('success', __('Construction activity deleted successfully.'));
        }
        return redirect()->back()->with('error', __('Permission Denied.'));
    }

    public function show($id)
    {
        if (Auth::user()->can('view project')) {
            $activity = ConstructionActivity::with(['stage.project', 'assignedUsers'])->findOrFail($id);

            return view('construction.activities.show', compact('activity'));
        }
        return redirect()->back()->with('error', __('Permission Denied.'));
    }

    public function updateStatus(Request $request, $id)
    {
        if (Auth::user()->can('edit project')) {
            $activity = ConstructionActivity::findOrFail($id);
            $activity->status = $request->status;

            if ($request->status == 'completed') {
                $activity->progress_percentage = 100;
            }

            $activity->save();

            return response()->json([
                'success' => true,
                'message' => __('Activity status updated successfully.')
            ]);
        }
        return response()->json(['error' => 'Permission Denied'], 403);
    }

    public function gantt($project_id)
    {
        if (Auth::user()->can('manage project')) {
            $project = Project::with(['constructionStages.activities'])->findOrFail($project_id);

            // Check if user has access to this project
            $usr = Auth::user();
            if ($usr->type == 'client') {
                if ($project->client_id != $usr->id) {
                    return redirect()->back()->with('error', __('Permission Denied.'));
                }
            } elseif ($project->created_by != $usr->creatorId()) {
                return redirect()->back()->with('error', __('Permission Denied.'));
            }

            // Prepare Gantt chart data
            $ganttData = [];
            foreach ($project->constructionStages as $stage) {
                foreach ($stage->activities as $activity) {
                    if ($activity->start_date && $activity->end_date) {
                        $ganttData[] = [
                            'id' => 'activity-' . $activity->id,
                            'name' => $activity->name,
                            'stage' => $stage->name,
                            'start' => $activity->start_date->format('Y-m-d'),
                            'end' => $activity->end_date->format('Y-m-d'),
                            'progress' => $activity->progress_percentage,
                            'status' => $activity->status,
                            'dependencies' => '',
                            'custom_class' => 'gantt-activity-' . $activity->status
                        ];
                    }
                }
            }

            return view('construction.gantt', compact('project', 'ganttData'));
        }
        return redirect()->back()->with('error', __('Permission Denied.'));
    }
}
