<?php

namespace App\Http\Controllers;

use App\Models\BudgetAlert;
use App\Models\Budget;
use App\Services\BudgetAlertService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class BudgetAlertController extends Controller
{
    protected $budgetAlertService;

    public function __construct(BudgetAlertService $budgetAlertService)
    {
        $this->budgetAlertService = $budgetAlertService;
    }

    /**
     * Display the budget alerts dashboard
     */
    public function index()
    {
        if (!Auth::user()->can('manage budget plan')) {
            return redirect()->back()->with('error', __('Permission denied.'));
        }

        $userId = Auth::user()->creatorId();
        
        // Get all active alerts for the user's budgets
        $activeAlerts = BudgetAlert::with('budget')
            ->whereHas('budget', function($query) use ($userId) {
                $query->where('created_by', $userId);
            })
            ->where('status', 'active')
            ->orderBy('alert_type', 'desc') // overrun first, then critical, then warning
            ->orderBy('current_percentage', 'desc')
            ->get();

        // Get alert statistics
        $alertStats = [
            'total' => $activeAlerts->count(),
            'overrun' => $activeAlerts->where('alert_type', 'overrun')->count(),
            'critical' => $activeAlerts->where('alert_type', 'critical')->count(),
            'warning' => $activeAlerts->where('alert_type', 'warning')->count(),
        ];

        // Get recent alerts (last 30 days)
        $recentAlerts = BudgetAlert::with('budget')
            ->whereHas('budget', function($query) use ($userId) {
                $query->where('created_by', $userId);
            })
            ->where('triggered_at', '>=', now()->subDays(30))
            ->orderBy('triggered_at', 'desc')
            ->limit(10)
            ->get();

        // Get budgets with their current status
        $budgets = Budget::where('created_by', $userId)->get();
        $budgetStatuses = [];
        
        foreach ($budgets as $budget) {
            $budgetAmount = $this->calculateBudgetAmount($budget);
            $spentAmount = $this->calculateSpentAmount($budget);
            $percentage = $budgetAmount > 0 ? ($spentAmount / $budgetAmount) * 100 : 0;
            
            $status = 'safe';
            if ($percentage >= 100) {
                $status = 'overrun';
            } elseif ($percentage >= 90) {
                $status = 'critical';
            } elseif ($percentage >= 75) {
                $status = 'warning';
            }
            
            $budgetStatuses[] = [
                'budget' => $budget,
                'percentage' => $percentage,
                'status' => $status,
                'budget_amount' => $budgetAmount,
                'spent_amount' => $spentAmount,
                'remaining_amount' => $budgetAmount - $spentAmount
            ];
        }

        return view('budget.alerts.index', compact('activeAlerts', 'alertStats', 'recentAlerts', 'budgetStatuses'));
    }

    /**
     * Acknowledge an alert
     */
    public function acknowledge(Request $request, $id)
    {
        if (!Auth::user()->can('manage budget plan')) {
            return response()->json(['error' => __('Permission denied.')], 403);
        }

        try {
            $alert = BudgetAlert::findOrFail($id);
            
            // Check if user owns this budget
            if ($alert->budget->created_by != Auth::user()->creatorId()) {
                return response()->json(['error' => __('Permission denied.')], 403);
            }

            $alert->acknowledge(Auth::id(), $request->input('notes'));

            Log::info("Budget alert acknowledged", [
                'alert_id' => $alert->id,
                'budget_id' => $alert->budget_id,
                'acknowledged_by' => Auth::id()
            ]);

            return response()->json([
                'success' => true,
                'message' => __('Alert acknowledged successfully.')
            ]);

        } catch (\Exception $e) {
            Log::error("Failed to acknowledge budget alert: " . $e->getMessage());
            return response()->json(['error' => __('Failed to acknowledge alert.')], 500);
        }
    }

    /**
     * Resolve an alert
     */
    public function resolve(Request $request, $id)
    {
        if (!Auth::user()->can('manage budget plan')) {
            return response()->json(['error' => __('Permission denied.')], 403);
        }

        try {
            $alert = BudgetAlert::findOrFail($id);
            
            // Check if user owns this budget
            if ($alert->budget->created_by != Auth::user()->creatorId()) {
                return response()->json(['error' => __('Permission denied.')], 403);
            }

            $alert->resolve(Auth::id(), $request->input('notes'));

            Log::info("Budget alert resolved", [
                'alert_id' => $alert->id,
                'budget_id' => $alert->budget_id,
                'resolved_by' => Auth::id()
            ]);

            return response()->json([
                'success' => true,
                'message' => __('Alert resolved successfully.')
            ]);

        } catch (\Exception $e) {
            Log::error("Failed to resolve budget alert: " . $e->getMessage());
            return response()->json(['error' => __('Failed to resolve alert.')], 500);
        }
    }

    /**
     * Get alert details
     */
    public function show($id)
    {
        if (!Auth::user()->can('manage budget plan')) {
            return response()->json(['error' => __('Permission denied.')], 403);
        }

        try {
            $alert = BudgetAlert::with(['budget', 'acknowledgedByUser', 'resolvedByUser'])
                ->findOrFail($id);
            
            // Check if user owns this budget
            if ($alert->budget->created_by != Auth::user()->creatorId()) {
                return response()->json(['error' => __('Permission denied.')], 403);
            }

            return response()->json([
                'success' => true,
                'alert' => $alert
            ]);

        } catch (\Exception $e) {
            Log::error("Failed to get budget alert details: " . $e->getMessage());
            return response()->json(['error' => __('Alert not found.')], 404);
        }
    }

    /**
     * Refresh alerts by running the check
     */
    public function refresh()
    {
        if (!Auth::user()->can('manage budget plan')) {
            return response()->json(['error' => __('Permission denied.')], 403);
        }

        try {
            $alertsCreated = $this->budgetAlertService->checkAllBudgets();
            
            return response()->json([
                'success' => true,
                'message' => __('Budget alerts refreshed successfully.'),
                'alerts_created' => $alertsCreated
            ]);

        } catch (\Exception $e) {
            Log::error("Failed to refresh budget alerts: " . $e->getMessage());
            return response()->json(['error' => __('Failed to refresh alerts.')], 500);
        }
    }

    /**
     * Calculate budget amount from income data
     */
    private function calculateBudgetAmount(Budget $budget)
    {
        $incomeData = json_decode($budget->income_data, true);
        $total = 0;

        if (is_array($incomeData)) {
            foreach ($incomeData as $income) {
                if (isset($income['amount'])) {
                    $total += (float) $income['amount'];
                }
            }
        }

        return $total;
    }

    /**
     * Calculate spent amount from expenses and bills
     */
    private function calculateSpentAmount(Budget $budget)
    {
        $startDate = $budget->from;
        $endDate = $budget->to;
        $createdBy = $budget->created_by;

        // Calculate expenses
        $expenseTotal = \App\Models\Expense::where('created_by', $createdBy)
            ->whereBetween('date', [$startDate, $endDate])
            ->sum('amount');

        // Calculate bills
        $billTotal = \App\Models\Bill::where('created_by', $createdBy)
            ->whereBetween('bill_date', [$startDate, $endDate])
            ->sum('total');

        return $expenseTotal + $billTotal;
    }
}