<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Services\BudgetAlertService;

class CheckBudgetAlerts extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'budget:check-alerts {--budget-id= : Check specific budget ID}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Check budget thresholds and create alerts for overruns';

    /**
     * The budget alert service instance.
     */
    protected $budgetAlertService;

    /**
     * Create a new command instance.
     */
    public function __construct(BudgetAlertService $budgetAlertService)
    {
        parent::__construct();
        $this->budgetAlertService = $budgetAlertService;
    }

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Starting budget alert monitoring...');

        try {
            $budgetId = $this->option('budget-id');

            if ($budgetId) {
                // Check specific budget
                $budget = \App\Models\Budget::findOrFail($budgetId);
                $this->info("Checking budget: {$budget->name}");
                $this->budgetAlertService->checkBudgetThresholds($budget);
                $this->info("Budget check completed for: {$budget->name}");
            } else {
                // Check all budgets
                $this->info('Checking all budgets...');
                $this->budgetAlertService->checkAllBudgets();
                $this->info('All budget checks completed.');
            }

            // Display summary of active alerts
            $activeAlerts = $this->budgetAlertService->getAllActiveAlerts();
            
            if ($activeAlerts->count() > 0) {
                $this->warn("Found {$activeAlerts->count()} active budget alerts:");
                
                $headers = ['Budget', 'Alert Type', 'Percentage', 'Amount Spent', 'Budget Total', 'Triggered At'];
                $rows = [];
                
                foreach ($activeAlerts as $alert) {
                    $rows[] = [
                        $alert->budget->name,
                        ucfirst($alert->alert_type),
                        number_format($alert->current_percentage, 1) . '%',
                        '$' . number_format($alert->spent_amount, 2),
                        '$' . number_format($alert->budget_amount, 2),
                        $alert->triggered_at->format('Y-m-d H:i:s')
                    ];
                }
                
                $this->table($headers, $rows);
            } else {
                $this->info('No active budget alerts found.');
            }

        } catch (\Exception $e) {
            $this->error('Error during budget alert check: ' . $e->getMessage());
            return 1;
        }

        return 0;
    }
}
