# Construction Management System - Database Schema

## Core Entities

### 1. Projects & Budget Planning
```sql
-- Projects
CREATE TABLE projects (
    id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(255) NOT NULL,
    description TEXT,
    client_id BIGINT UNSIGNED,
    start_date DATE,
    end_date DATE,
    status ENUM('planning', 'active', 'completed', 'cancelled') DEFAULT 'planning',
    total_budget DECIMAL(15,2),
    created_at TIMESTAMP,
    updated_at TIMESTAMP,
    FOREIGN KEY (client_id) REFERENCES clients(id)
);

-- Project Stages (Foundation, Slab, Plastering, etc.)
CREATE TABLE project_stages (
    id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
    project_id BIGINT UNSIGNED,
    name VARCHAR(255) NOT NULL,
    description TEXT,
    budget_allocated DECIMAL(15,2),
    start_date DATE,
    end_date DATE,
    status ENUM('pending', 'in_progress', 'completed') DEFAULT 'pending',
    sequence_order INT,
    created_at TIMESTAMP,
    updated_at TIMESTAMP,
    FOREIGN KEY (project_id) REFERENCES projects(id) ON DELETE CASCADE
);

-- Activities within each stage
CREATE TABLE project_activities (
    id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
    stage_id BIGINT UNSIGNED,
    name VARCHAR(255) NOT NULL,
    description TEXT,
    estimated_cost DECIMAL(15,2),
    actual_cost DECIMAL(15,2) DEFAULT 0,
    labor_hours DECIMAL(8,2),
    material_cost DECIMAL(15,2) DEFAULT 0,
    status ENUM('pending', 'in_progress', 'completed', 'rework') DEFAULT 'pending',
    created_at TIMESTAMP,
    updated_at TIMESTAMP,
    FOREIGN KEY (stage_id) REFERENCES project_stages(id) ON DELETE CASCADE
);

-- Rate Contracts (Preloaded rates for different activities)
CREATE TABLE rate_contracts (
    id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
    activity_name VARCHAR(255) NOT NULL,
    unit VARCHAR(50), -- per sqft, per cubic meter, etc.
    base_rate DECIMAL(10,2),
    labor_component DECIMAL(10,2),
    material_component DECIMAL(10,2),
    effective_from DATE,
    effective_to DATE,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP,
    updated_at TIMESTAMP
);
```

### 2. Financial Management
```sql
-- Quotations
CREATE TABLE quotations (
    id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
    project_id BIGINT UNSIGNED,
    quotation_number VARCHAR(100) UNIQUE,
    version INT DEFAULT 1,
    total_amount DECIMAL(15,2),
    tax_amount DECIMAL(15,2),
    profit_margin_percentage DECIMAL(5,2),
    contingency_percentage DECIMAL(5,2),
    escalation_percentage DECIMAL(5,2),
    status ENUM('draft', 'sent', 'approved', 'rejected') DEFAULT 'draft',
    valid_until DATE,
    notes TEXT,
    created_at TIMESTAMP,
    updated_at TIMESTAMP,
    FOREIGN KEY (project_id) REFERENCES projects(id)
);

-- Payment Milestones
CREATE TABLE payment_milestones (
    id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
    quotation_id BIGINT UNSIGNED,
    stage_id BIGINT UNSIGNED,
    milestone_name VARCHAR(255),
    percentage DECIMAL(5,2),
    amount DECIMAL(15,2),
    due_date DATE,
    status ENUM('pending', 'invoiced', 'paid') DEFAULT 'pending',
    created_at TIMESTAMP,
    updated_at TIMESTAMP,
    FOREIGN KEY (quotation_id) REFERENCES quotations(id),
    FOREIGN KEY (stage_id) REFERENCES project_stages(id)
);

-- Expenses
CREATE TABLE expenses (
    id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
    project_id BIGINT UNSIGNED,
    activity_id BIGINT UNSIGNED NULL,
    expense_type ENUM('material', 'manpower', 'third_party', 'petty_cash', 'asset') NOT NULL,
    category VARCHAR(100),
    description TEXT,
    amount DECIMAL(15,2),
    expense_date DATE,
    receipt_number VARCHAR(100),
    approved_by BIGINT UNSIGNED NULL,
    status ENUM('pending', 'approved', 'rejected') DEFAULT 'pending',
    created_at TIMESTAMP,
    updated_at TIMESTAMP,
    FOREIGN KEY (project_id) REFERENCES projects(id),
    FOREIGN KEY (activity_id) REFERENCES project_activities(id),
    FOREIGN KEY (approved_by) REFERENCES users(id)
);
```

### 3. Inventory Management
```sql
-- Vendors
CREATE TABLE vendors (
    id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(255) NOT NULL,
    contact_person VARCHAR(255),
    phone VARCHAR(20),
    email VARCHAR(255),
    address TEXT,
    gst_number VARCHAR(50),
    payment_terms VARCHAR(100),
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP,
    updated_at TIMESTAMP
);

-- Materials
CREATE TABLE materials (
    id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(255) NOT NULL,
    category VARCHAR(100),
    unit VARCHAR(50), -- kg, bags, pieces, etc.
    current_rate DECIMAL(10,2),
    min_stock_level INT DEFAULT 0,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP,
    updated_at TIMESTAMP
);

-- Material Inventory
CREATE TABLE material_inventory (
    id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
    material_id BIGINT UNSIGNED,
    quantity_in_stock DECIMAL(10,2) DEFAULT 0,
    reserved_quantity DECIMAL(10,2) DEFAULT 0,
    last_purchase_rate DECIMAL(10,2),
    last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (material_id) REFERENCES materials(id)
);

-- Purchase Orders
CREATE TABLE purchase_orders (
    id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
    vendor_id BIGINT UNSIGNED,
    po_number VARCHAR(100) UNIQUE,
    order_date DATE,
    expected_delivery_date DATE,
    total_amount DECIMAL(15,2),
    status ENUM('draft', 'sent', 'confirmed', 'delivered', 'cancelled') DEFAULT 'draft',
    created_by BIGINT UNSIGNED,
    created_at TIMESTAMP,
    updated_at TIMESTAMP,
    FOREIGN KEY (vendor_id) REFERENCES vendors(id),
    FOREIGN KEY (created_by) REFERENCES users(id)
);

-- Purchase Order Items
CREATE TABLE purchase_order_items (
    id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
    po_id BIGINT UNSIGNED,
    material_id BIGINT UNSIGNED,
    quantity DECIMAL(10,2),
    unit_rate DECIMAL(10,2),
    total_amount DECIMAL(15,2),
    received_quantity DECIMAL(10,2) DEFAULT 0,
    created_at TIMESTAMP,
    updated_at TIMESTAMP,
    FOREIGN KEY (po_id) REFERENCES purchase_orders(id) ON DELETE CASCADE,
    FOREIGN KEY (material_id) REFERENCES materials(id)
);

-- Material Issues to Projects
CREATE TABLE material_issues (
    id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
    project_id BIGINT UNSIGNED,
    activity_id BIGINT UNSIGNED,
    material_id BIGINT UNSIGNED,
    quantity_issued DECIMAL(10,2),
    unit_rate DECIMAL(10,2),
    profit_margin_percentage DECIMAL(5,2) DEFAULT 0,
    total_cost DECIMAL(15,2),
    issue_date DATE,
    issued_by BIGINT UNSIGNED,
    created_at TIMESTAMP,
    updated_at TIMESTAMP,
    FOREIGN KEY (project_id) REFERENCES projects(id),
    FOREIGN KEY (activity_id) REFERENCES project_activities(id),
    FOREIGN KEY (material_id) REFERENCES materials(id),
    FOREIGN KEY (issued_by) REFERENCES users(id)
);
```

### 4. Asset Management
```sql
-- Assets
CREATE TABLE assets (
    id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(255) NOT NULL,
    asset_code VARCHAR(100) UNIQUE,
    category VARCHAR(100),
    purchase_cost DECIMAL(15,2),
    current_value DECIMAL(15,2),
    depreciation_rate DECIMAL(5,2),
    hourly_rate DECIMAL(10,2),
    fuel_cost_per_hour DECIMAL(10,2) DEFAULT 0,
    status ENUM('available', 'in_use', 'maintenance', 'retired') DEFAULT 'available',
    created_at TIMESTAMP,
    updated_at TIMESTAMP
);

-- Asset Allotments
CREATE TABLE asset_allotments (
    id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
    asset_id BIGINT UNSIGNED,
    project_id BIGINT UNSIGNED,
    start_date DATE,
    end_date DATE,
    hours_used DECIMAL(8,2) DEFAULT 0,
    total_cost DECIMAL(15,2) DEFAULT 0,
    status ENUM('active', 'completed') DEFAULT 'active',
    created_at TIMESTAMP,
    updated_at TIMESTAMP,
    FOREIGN KEY (asset_id) REFERENCES assets(id),
    FOREIGN KEY (project_id) REFERENCES projects(id)
);
```

### 5. User Management & Petty Cash
```sql
-- Users (Extended from Laravel's default)
CREATE TABLE user_profiles (
    id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
    user_id BIGINT UNSIGNED,
    employee_code VARCHAR(100),
    department VARCHAR(100),
    designation VARCHAR(100),
    petty_cash_limit DECIMAL(10,2) DEFAULT 0,
    current_petty_cash_balance DECIMAL(10,2) DEFAULT 0,
    created_at TIMESTAMP,
    updated_at TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);

-- Petty Cash Transactions
CREATE TABLE petty_cash_transactions (
    id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
    user_id BIGINT UNSIGNED,
    project_id BIGINT UNSIGNED,
    activity_id BIGINT UNSIGNED NULL,
    transaction_type ENUM('received', 'spent', 'claim') NOT NULL,
    amount DECIMAL(10,2),
    description TEXT,
    receipt_number VARCHAR(100),
    transaction_date DATE,
    approved_by BIGINT UNSIGNED NULL,
    status ENUM('pending', 'approved', 'rejected') DEFAULT 'pending',
    created_at TIMESTAMP,
    updated_at TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id),
    FOREIGN KEY (project_id) REFERENCES projects(id),
    FOREIGN KEY (activity_id) REFERENCES project_activities(id),
    FOREIGN KEY (approved_by) REFERENCES users(id)
);
```

### 6. Accounting Module
```sql
-- Chart of Accounts
CREATE TABLE chart_of_accounts (
    id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
    account_code VARCHAR(20) UNIQUE,
    account_name VARCHAR(255) NOT NULL,
    account_type ENUM('asset', 'liability', 'equity', 'income', 'expense') NOT NULL,
    parent_account_id BIGINT UNSIGNED NULL,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP,
    updated_at TIMESTAMP,
    FOREIGN KEY (parent_account_id) REFERENCES chart_of_accounts(id)
);

-- Journal Entries
CREATE TABLE journal_entries (
    id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
    entry_number VARCHAR(100) UNIQUE,
    entry_date DATE,
    description TEXT,
    reference_type VARCHAR(50), -- expense, purchase, payment, etc.
    reference_id BIGINT UNSIGNED,
    total_debit DECIMAL(15,2),
    total_credit DECIMAL(15,2),
    created_by BIGINT UNSIGNED,
    created_at TIMESTAMP,
    updated_at TIMESTAMP,
    FOREIGN KEY (created_by) REFERENCES users(id)
);

-- Journal Entry Details
CREATE TABLE journal_entry_details (
    id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
    journal_entry_id BIGINT UNSIGNED,
    account_id BIGINT UNSIGNED,
    debit_amount DECIMAL(15,2) DEFAULT 0,
    credit_amount DECIMAL(15,2) DEFAULT 0,
    description TEXT,
    created_at TIMESTAMP,
    updated_at TIMESTAMP,
    FOREIGN KEY (journal_entry_id) REFERENCES journal_entries(id) ON DELETE CASCADE,
    FOREIGN KEY (account_id) REFERENCES chart_of_accounts(id)
);
```

### 7. Supporting Tables
```sql
-- Clients
CREATE TABLE clients (
    id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(255) NOT NULL,
    contact_person VARCHAR(255),
    phone VARCHAR(20),
    email VARCHAR(255),
    address TEXT,
    gst_number VARCHAR(50),
    created_at TIMESTAMP,
    updated_at TIMESTAMP
);

-- Budget Alerts Configuration
CREATE TABLE budget_alerts (
    id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
    alert_type ENUM('project_budget', 'activity_budget', 'petty_cash') NOT NULL,
    threshold_percentage DECIMAL(5,2) DEFAULT 80.00,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP,
    updated_at TIMESTAMP
);
```

## Key Features of This Schema

1. **Hierarchical Structure**: Projects → Stages → Activities
2. **Flexible Expense Tracking**: Multiple expense types with proper categorization
3. **Comprehensive Inventory**: From purchase to project allocation
4. **Asset Management**: With depreciation and hourly cost tracking
5. **Double Entry Accounting**: Automated journal entries
6. **Alert System**: Configurable budget overrun alerts
7. **Audit Trail**: All tables have timestamps and proper foreign key relationships
8. **Scalability**: Designed to handle multiple projects simultaneously

## Laravel Model Relationships

The schema supports these key relationships:
- Project hasMany Stages hasMany Activities
- Project hasMany Expenses, MaterialIssues, AssetAllotments
- Material hasMany PurchaseOrderItems, MaterialIssues
- User hasMany PettyCashTransactions, Expenses (as approver)
- Vendor hasMany PurchaseOrders