# Construction Management System - Complete Scope of Work

## Project Overview

A comprehensive construction project management system built with **Laravel 12** backend and **Creative Tim's Material Angular** frontend. The system manages project budgeting, expense tracking, inventory, assets, and accounting with real-time alerts and reporting.

## Tech Stack & Design Requirements

### Backend
- **Framework**: Laravel 12
- **Database**: MySQL 8.0+
- **API**: RESTful APIs with JSON responses
- **Authentication**: Laravel Sanctum for SPA authentication
- **Queue System**: Redis/Database queues for background jobs
- **Storage**: Local/S3 for file uploads

### Frontend
- **Framework**: Angular with Creative Tim's Material Dashboard
- **UI Components**: **MUST use Material Design components from the Creative Tim template**
- **State Management**: Angular services with RxJS
- **HTTP Client**: Angular HttpClient
- **Authentication**: JWT token-based authentication
- **Charts**: ng2-charts or Chart.js integration within Material cards

### Design Philosophy
- **Material Design Principles**: Follow Google's Material Design guidelines
- **Responsive Design**: Mobile-first approach using Material breakpoints
- **Consistency**: Use only Material components from Creative Tim template
- **Color Scheme**: Primary (blue), secondary (gray), accent colors from Material theme
- **Typography**: Material typography scale
- **Icons**: Material Icons only

## Module Breakdown & Development Sequence

### Phase 1: Foundation & Core Project Management

#### 1.1 Database Schema
```sql
-- Core tables for project hierarchy
- projects
- project_stages  
- project_activities
- rate_contracts
- clients
- users (extend Laravel default)
- user_profiles
```

#### 1.2 Backend APIs
**Authentication & Users**
- `POST /api/auth/login` - User authentication
- `POST /api/auth/logout` - User logout
- `GET /api/user/profile` - Get user profile with petty cash info
- `PUT /api/user/profile` - Update user profile

**Projects Management**
- `GET /api/projects` - List all projects with pagination, filters
- `POST /api/projects` - Create new project
- `GET /api/projects/{id}` - Get project details with stages and activities
- `PUT /api/projects/{id}` - Update project
- `DELETE /api/projects/{id}` - Soft delete project
- `GET /api/projects/{id}/budget-summary` - Get budget vs actual costs

**Project Stages**
- `GET /api/projects/{id}/stages` - Get all stages for a project
- `POST /api/projects/{id}/stages` - Create new stage
- `PUT /api/stages/{id}` - Update stage
- `DELETE /api/stages/{id}` - Delete stage

**Project Activities**
- `GET /api/stages/{id}/activities` - Get activities for a stage
- `POST /api/stages/{id}/activities` - Create new activity
- `PUT /api/activities/{id}` - Update activity
- `POST /api/activities/{id}/rework` - Mark activity for rework

**Rate Contracts**
- `GET /api/rate-contracts` - Get active rate contracts
- `POST /api/rate-contracts` - Create rate contract
- `PUT /api/rate-contracts/{id}` - Update rate contract

#### 1.3 Frontend Components (Material Design)

**Dashboard Layout**
- Use Material sidenav with project tree navigation
- Material toolbar with user menu and notifications
- Material cards for KPI widgets (total projects, budget utilization, etc.)

**Projects Module**
- **Project List**: Material data table with sorting, pagination, filtering
- **Project Form**: Material form fields (input, select, datepicker, textarea)
- **Project Detail**: Material tabs for Overview, Stages, Budget, Expenses
- **Stage Management**: Material expansion panels for stage hierarchy
- **Activity Tracking**: Material progress bars and status chips

**Key Material Components to Use:**
- `mat-card` for all content containers
- `mat-table` for data lists
- `mat-form-field` with `mat-input` for forms
- `mat-select` for dropdowns
- `mat-datepicker` for date inputs
- `mat-button` and `mat-fab` for actions
- `mat-chip` for status indicators
- `mat-progress-bar` for progress tracking
- `mat-expansion-panel` for collapsible content
- `mat-tabs` for section navigation

### Phase 2: Financial Management & Quotations

#### 2.1 Database Schema
```sql
- quotations
- payment_milestones
- expenses
- budget_alerts
```

#### 2.2 Backend APIs
**Quotations**
- `GET /api/projects/{id}/quotations` - Get project quotations
- `POST /api/projects/{id}/quotations` - Create quotation
- `PUT /api/quotations/{id}` - Update quotation
- `POST /api/quotations/{id}/duplicate` - Create new version
- `POST /api/quotations/{id}/send` - Send to client
- `GET /api/quotations/{id}/pdf` - Generate PDF

**Payment Milestones**
- `GET /api/quotations/{id}/milestones` - Get payment milestones
- `POST /api/quotations/{id}/milestones` - Create milestone
- `PUT /api/milestones/{id}/mark-paid` - Mark as paid

**Expense Tracking**
- `GET /api/projects/{id}/expenses` - Get project expenses (filterable)
- `POST /api/projects/{id}/expenses` - Create expense
- `PUT /api/expenses/{id}` - Update expense
- `POST /api/expenses/{id}/approve` - Approve expense
- `GET /api/expenses/pending-approval` - Pending approvals

**Budget Alerts**
- `GET /api/alerts/budget-overruns` - Get budget alert notifications
- `POST /api/alerts/mark-read` - Mark alerts as read

#### 2.3 Frontend Components

**Quotation Management**
- Material stepper for quotation creation workflow
- Material table for quotation versions with action menu
- Material dialog for quotation preview
- Material form with calculated fields (auto-compute totals)

**Expense Tracking**
- Material data source with real-time filtering
- Material form with expense type chips
- Material date range picker for expense reports
- Material progress indicators for budget utilization

**Financial Dashboard**
- Material cards with Chart.js integration
- Material tabs for different report views
- Color-coded Material chips for budget status (green/yellow/red)

### Phase 3: Inventory Management

#### 3.1 Database Schema
```sql
- vendors
- materials
- material_inventory
- purchase_orders
- purchase_order_items
- material_issues
```

#### 3.2 Backend APIs
**Vendors**
- `GET /api/vendors` - List vendors with search
- `POST /api/vendors` - Create vendor
- `PUT /api/vendors/{id}` - Update vendor
- `GET /api/vendors/{id}/purchase-history` - Purchase history

**Materials**
- `GET /api/materials` - List materials with stock levels
- `POST /api/materials` - Create material
- `PUT /api/materials/{id}` - Update material
- `GET /api/materials/low-stock` - Low stock alerts

**Purchase Orders**
- `GET /api/purchase-orders` - List POs with filters
- `POST /api/purchase-orders` - Create PO
- `PUT /api/purchase-orders/{id}` - Update PO
- `POST /api/purchase-orders/{id}/send` - Send to vendor
- `POST /api/purchase-orders/{id}/receive` - Mark items as received

**Material Issues**
- `GET /api/projects/{id}/material-issues` - Project material usage
- `POST /api/projects/{id}/material-issues` - Issue materials to project
- `GET /api/materials/{id}/usage-history` - Material usage across projects

#### 3.3 Frontend Components

**Inventory Dashboard**
- Material cards showing stock levels with color indicators
- Material table with stock alert badges
- Material charts for usage trends

**Purchase Order Management**
- Material stepper for PO creation
- Material table with expandable rows for PO items
- Material dialog for receiving goods
- Material autocomplete for vendor/material selection

**Material Issue Interface**
- Material select with real-time stock checking
- Material input with quantity validation
- Material confirmation dialog with cost calculations

### Phase 4: Asset Management

#### 4.1 Database Schema
```sql
- assets
- asset_allotments
```

#### 4.2 Backend APIs
**Assets**
- `GET /api/assets` - List assets with availability status
- `POST /api/assets` - Create asset
- `PUT /api/assets/{id}` - Update asset
- `GET /api/assets/{id}/usage-history` - Asset usage history

**Asset Allotments**
- `GET /api/projects/{id}/asset-allotments` - Project asset usage
- `POST /api/projects/{id}/asset-allotments` - Allocate asset to project
- `PUT /api/asset-allotments/{id}/complete` - Complete asset usage
- `GET /api/assets/availability` - Check asset availability

#### 4.3 Frontend Components
- Material cards for asset overview with status indicators
- Material calendar view for asset scheduling
- Material form for asset allocation with cost preview
- Material table with asset utilization metrics

### Phase 5: Petty Cash Management

#### 5.1 Database Schema
```sql
- petty_cash_transactions
```

#### 5.2 Backend APIs
- `GET /api/user/petty-cash/balance` - Current balance
- `POST /api/user/petty-cash/claim` - Submit expense claim
- `GET /api/user/petty-cash/transactions` - Transaction history
- `GET /api/petty-cash/pending-approvals` - For managers
- `POST /api/petty-cash/approve/{id}` - Approve claim

#### 5.3 Frontend Components
- Material card showing current balance and limit
- Material form for expense claims with receipt upload
- Material table for transaction history
- Material badge notifications for pending approvals

### Phase 6: Accounting Module

#### 6.1 Database Schema
```sql
- chart_of_accounts
- journal_entries
- journal_entry_details
```

#### 6.2 Backend APIs
**Chart of Accounts**
- `GET /api/accounts` - Get chart of accounts tree
- `POST /api/accounts` - Create account
- `PUT /api/accounts/{id}` - Update account

**Journal Entries**
- `GET /api/journal-entries` - List entries with filters
- `POST /api/journal-entries` - Manual journal entry
- `GET /api/reports/profit-loss` - P&L report by project
- `GET /api/reports/balance-sheet` - Balance sheet
- `GET /api/reports/cash-flow` - Cash flow report

#### 6.3 Frontend Components
- Material tree for chart of accounts
- Material table for journal entries with drill-down
- Material cards for financial reports with charts
- Material date range picker for report periods

## UI/UX Guidelines

### Material Design Implementation

#### Layout Structure
```typescript
// App component should use Material sidenav container
<mat-sidenav-container>
  <mat-sidenav #drawer mode="side" opened>
    // Navigation menu using mat-nav-list
  </mat-sidenav>
  <mat-sidenav-content>
    <mat-toolbar color="primary">
      // Top navigation
    </mat-toolbar>
    <div class="main-content">
      <router-outlet></router-outlet>
    </div>
  </mat-sidenav-content>
</mat-sidenav-container>
```

#### Component Patterns
1. **List Views**: Always use `mat-table` with `mat-paginator`
2. **Forms**: Use `mat-form-field` with proper validation
3. **Actions**: Primary actions use `mat-raised-button`, secondary use `mat-button`
4. **Status Display**: Use `mat-chip` with appropriate colors
5. **Data Cards**: Use `mat-card` with `mat-card-header`, `mat-card-content`
6. **Navigation**: Use `mat-tabs` for section switching

#### Color Scheme
- **Primary**: Blue (#1976D2) - for main actions and headers
- **Accent**: Orange (#FF5722) - for important alerts and CTAs  
- **Warn**: Red (#F44336) - for errors and critical alerts
- **Success**: Green (#4CAF50) - for success states and positive indicators

#### Responsive Breakpoints
- Mobile: `mat-grid-tile` with colspan based on screen size
- Tablet: Adjust `mat-sidenav` mode to `over`
- Desktop: Full sidebar navigation with `side` mode

### Data Display Standards

#### Tables
```html
<mat-card>
  <mat-card-header>
    <mat-card-title>Projects</mat-card-title>
  </mat-card-header>
  <mat-card-content>
    <mat-table [dataSource]="dataSource" matSort>
      <!-- columns definition -->
    </mat-table>
    <mat-paginator [pageSizeOptions]="[10, 25, 50]"></mat-paginator>
  </mat-card-content>
</mat-card>
```

#### Forms
```html
<mat-card>
  <mat-card-header>
    <mat-card-title>Create Project</mat-card-title>
  </mat-card-header>
  <mat-card-content>
    <form [formGroup]="projectForm">
      <mat-form-field appearance="outline" class="full-width">
        <mat-label>Project Name</mat-label>
        <input matInput formControlName="name" required>
        <mat-error>Project name is required</mat-error>
      </mat-form-field>
      <!-- other form fields -->
    </form>
  </mat-card-content>
  <mat-card-actions align="end">
    <button mat-button type="button">Cancel</button>
    <button mat-raised-button color="primary" type="submit">Save</button>
  </mat-card-actions>
</mat-card>
```

## Development Workflow for Claude Code

### Step 1: Setup Project Structure
```bash
claude-code "Based on the scope document, create the initial Laravel project structure with proper folder organization for Controllers, Models, Services, and Requests. Set up the database configuration and create the base migration files."
```

### Step 2: Create Database Migrations
```bash
claude-code "Create Laravel migrations for the Phase 1 tables (projects, project_stages, project_activities, rate_contracts, clients, user_profiles) following the exact schema specifications in the scope document."
```

### Step 3: Build Eloquent Models
```bash
claude-code "Create Eloquent models with proper relationships, fillable fields, casts, and accessor/mutator methods for the Phase 1 entities. Include validation rules and implement soft deletes where needed."
```

### Step 4: Develop API Controllers
```bash
claude-code "Create API controllers for the Phase 1 endpoints following RESTful conventions. Include proper validation, error handling, and JSON responses. Implement resource transformers for consistent API output."
```

### Step 5: Frontend Components
```bash
claude-code "Create Angular components for the Projects module using ONLY Material Design components from Creative Tim template. Implement the project list, create/edit forms, and detail views following the Material Design patterns specified in the scope."
```

### Step 6: Integration & Testing
```bash
claude-code "Integrate the frontend and backend, implement proper error handling, loading states, and form validations using Material components. Create API service methods and implement state management."
```

## Quality Standards

### Backend Requirements
- Follow Laravel coding standards and PSR-12
- Implement proper validation using Form Requests
- Use Eloquent ORM relationships efficiently
- Implement proper error handling and logging
- Add API documentation with clear examples
- Include unit and feature tests

### Frontend Requirements
- **STRICT**: Use only Material Design components from Creative Tim template
- Follow Angular style guide and best practices
- Implement proper TypeScript interfaces
- Use RxJS operators efficiently
- Add loading states and error handling to all components
- Ensure responsive design across all screen sizes
- Maintain consistent Material Design visual hierarchy

### Performance Requirements
- API responses under 200ms for standard queries
- Frontend initial load under 3 seconds
- Implement proper caching strategies
- Use database indexing appropriately
- Implement pagination for large datasets

This scope document should guide Claude Code through each development phase while ensuring consistency with Material Design principles and maintaining high code quality standards.