# Template System — Design & Implementation Plan

## Goal
Allow companies to save project templates (predefined Stages → Activities → Items/Materials + default quantity rules and rates) and apply them to any new project, with automatic quantity scaling based on project area or configurable parameters — minimizing repeated data entry.

---

## 1) Concepts & Entities
- **Template** — project type (e.g., Duplex 3BHK, Apartment 2BHK).
- **Template Stage** — Foundation, Superstructure, Finishing, etc.
- **Template Activity** — e.g., Column Casting, Brickwork.
- **Template Item (BillItem)** — material/labour with quantity rule.
- **Parameter** — inputs for quantity rules (area, floors, rooms).
- **AppliedTemplate / ProjectTemplateInstance** — snapshot when applied.
- **TemplateVersion** — versioned templates.

---

## 2) Database Schema
```sql
CREATE TABLE templates (...);
CREATE TABLE template_stages (...);
CREATE TABLE template_activities (...);
CREATE TABLE template_items (...);
CREATE TABLE project_template_instances (...);
CREATE TABLE project_template_items (...);
```
- Stores templates, stages, activities, items with rules.
- `quantity_rule` is expression string (safe evaluated).
- Snapshot stored in project_template_items.

---

## 3) Quantity Rule Algorithm
- Use safe expression evaluator (Symfony ExpressionLanguage or custom).
- Parameters: `area`, `area_sqft`, `floors`, `rooms`.
- Example rules:
  - Cement: `ceil(area_sqft * 0.02)`
  - Steel: `ceil(area_sqft * floors * 0.02)`
  - Bricks: `ceil(area_sqft * 50)`

### Example Evaluation
Project area = 100 sq yd = 900 sq ft.
- Cement: 18 bags
- Steel: 36 kg
- Bricks: 45,000 nos

---

## 4) Frontend UX Flows
### Template Library
- `/templates` route.
- Create/Edit/Clone/Version.
- `mat-stepper` wizard: Metadata → Stages → Activities → Items.

### Apply Template to Project
- In Create Project page: select template → enter params (area, floors).
- Preview summary (materials + budget).
- Apply: creates stages/activities + snapshot.

### Project-level Adjustments
- Inline edit resolved quantities.
- Bulk edit via CSV.

### Bulk Apply
- Upload CSV with params (project_id, area, floors).

---

## 5) Backend APIs
```
GET  /api/templates
POST /api/templates
GET  /api/templates/{id}
PUT  /api/templates/{id}
POST /api/templates/{id}/clone
POST /api/templates/{id}/version
GET  /api/templates/{id}/preview?area=900&floors=2

POST /api/projects/{project_id}/apply-template
GET  /api/projects/{project_id}/template-instance

POST /api/templates/bulk-apply
POST /api/templates/import
GET  /api/templates/export/{id}
```

---

## 6) Seeding & Onboarding
- Preload Duplex, Apartment, Villa templates.
- Defaults based on UltraTech & Estimation reports.

---

## 7) Versioning & Governance
- Increment version on edits.
- Old projects keep their snapshot.
- Publish templates to company library.

---

## 8) UX Principles to Minimize Data Entry
- Defaults for params.
- Param-based rules.
- Preview before applying.
- Clone + adjust instead of re-entry.
- Bulk apply via CSV.
- Quick mobile forms.

---

## 9) Example Template JSON
```json
{
  "name": "Duplex 3BHK (100 sqyd prescale)",
  "unit_default": "sq_yd",
  "default_params": { "area": 100, "floors": 2 },
  "stages": [
    {
      "title": "Foundation",
      "activities": [
        {
          "title": "Excavation & Footings",
          "items": [
            { "title": "Cement bags", "unit": "bags", "quantity_rule": "ceil(area_sqft * 0.02)", "unit_rate": 450 },
            { "title": "Steel - kg", "unit": "kg", "quantity_rule": "ceil(area_sqft * floors * 0.02)", "unit_rate": 75 }
          ]
        }
      ]
    },
    {
      "title": "Superstructure",
      "activities": [
        {
          "title": "Brickwork",
          "items": [
            { "title": "Bricks", "unit": "nos", "quantity_rule": "ceil(area_sqft * 50)", "unit_rate": 6 }
          ]
        }
      ]
    }
  ]
}
```

---

## 10) Acceptance Criteria
- Max 3 inputs when applying (area, floors, plinth).
- Preview shown before apply.
- Applied snapshot is immutable.
- Bulk apply works with CSV.
- Rule tester validates outputs.

---

## 11) Roadmap (20 days)
- **Day 1–3**: DB migrations, CRUD.
- **Day 4–7**: Preview + apply endpoints.
- **Day 8–12**: Angular UI (library, apply modal).
- **Day 13–16**: Bulk apply + rule tester.
- **Day 17–20**: QA + sample seed templates.

---

## 12) Future Enhancements
- CAD/BIM integrations.
- Material catalog mapping.
- Unit localization (sq_m, sq_ft).
- Marketplace for shared templates.
- Audit logs on template changes.

