# Construction ERP System - Technical Documentation

## Table of Contents
1. [System Architecture](#system-architecture)
2. [Database Schema](#database-schema)
3. [API Documentation](#api-documentation)
4. [Frontend Architecture](#frontend-architecture)
5. [Authentication & Security](#authentication--security)
6. [Deployment Guide](#deployment-guide)
7. [Development Setup](#development-setup)
8. [Testing Guidelines](#testing-guidelines)
9. [Performance Optimization](#performance-optimization)
10. [Troubleshooting](#troubleshooting)

---

## System Architecture

### Backend Architecture (Laravel 12)
```
├── app/
│   ├── Http/
│   │   ├── Controllers/     # API Controllers
│   │   ├── Middleware/      # Custom Middleware
│   │   ├── Requests/        # Form Validation
│   │   └── Resources/       # API Resources
│   ├── Models/              # Eloquent Models
│   ├── Services/            # Business Logic
│   └── Providers/           # Service Providers
├── database/
│   ├── migrations/           # Database Migrations
│   └── seeders/             # Database Seeders
├── routes/
│   └── api.php              # API Routes
└── config/                  # Configuration Files
```

### Frontend Architecture (Laravel Blade + Vite)
```
├── resources/
│   ├── views/
│   │   ├── layouts/          # Base layouts (e.g., app.blade.php)
│   │   ├── partials/         # Shared partials (header, nav, footer)
│   │   ├── components/       # Reusable Blade components
│   │   └── pages/            # Feature-specific views
│   ├── css/                  # Styles compiled via Vite
│   ├── js/                   # ES modules compiled via Vite
│   └── lang/                 # i18n strings
├── public/
│   ├── assets/               # Images, fonts
│   ├── css/                  # Built styles (if output to public)
│   └── js/                   # Built scripts (if output to public)
└── vite.config.js            # Vite build configuration
```

#### Blade Conventions
- Use `resources/views/layouts/app.blade.php` as the base layout and include assets via `@vite(['resources/js/app.js', 'resources/css/app.css'])`.
- Create reusable UI with Blade components under `resources/views/components` and include via `<x-component-name />`.
- Structure feature pages under `resources/views/pages/{feature}` and render in controllers with `return view('pages.feature.index', $data);`.
- For interactivity, progressively enhance with lightweight JS (e.g., Alpine.js or vanilla modules) compiled by Vite; avoid SPA complexity.

---

## Database Schema

### Core Tables

#### Users & Authentication
```sql
-- users (Laravel default)
id, name, email, email_verified_at, password, remember_token, created_at, updated_at

-- user_profiles
id, user_id, employee_code, department, designation, phone, address, hire_date, petty_cash_limit, current_petty_cash_balance

-- roles
id, name, display_name, description, is_active, created_at, updated_at

-- permissions
id, name, display_name, description, module, is_active, created_at, updated_at

-- role_user (pivot)
role_id, user_id

-- permission_role (pivot)
permission_id, role_id
```

#### Project Management
```sql
-- projects
id, name, description, client_id, start_date, end_date, status, total_budget, spent_amount, created_at, updated_at

-- project_stages
id, project_id, stage_name, description, sequence_order, estimated_cost, actual_cost, start_date, end_date, status, created_at, updated_at

-- project_activities
id, stage_id, name, description, estimated_cost, actual_cost, labor_hours, material_cost, status, created_at, updated_at

-- rate_contracts
id, name, description, rate_type, rate_value, unit, is_active, created_at, updated_at
```

#### Client Management
```sql
-- clients
id, type, name, company_name, contact_person, email, phone, alternate_phone, address, city, state, country, pincode, gst_number, pan_number, website, notes, status, credit_limit, outstanding_amount, registration_date, last_contact_date, created_at, updated_at
```

#### Financial Management
```sql
-- quotations
id, project_id, quotation_number, version, status, valid_until, subtotal, tax_rate, tax_amount, discount_percentage, discount_amount, total_amount, terms_conditions, notes, client_signature, sent_at, approved_at, expires_at, created_at, updated_at

-- payment_milestones
id, quotation_id, milestone_name, description, amount, percentage, due_date, status, paid_at, payment_reference, notes, created_at, updated_at

-- expenses
id, project_id, user_id, expense_number, title, description, category, amount, expense_date, receipt_path, status, approved_by, approved_at, rejection_reason, notes, created_at, updated_at

-- budget_alerts
id, project_id, alert_type, title, message, severity, status, alert_data, created_by, acknowledged_by, acknowledged_at, resolved_at, created_at, updated_at
```

### Relationships
- **Projects** → **Clients** (belongsTo)
- **Projects** → **Stages** (hasMany)
- **Stages** → **Activities** (hasMany)
- **Projects** → **Quotations** (hasMany)
- **Projects** → **Expenses** (hasMany)
- **Projects** → **Budget Alerts** (hasMany)
- **Quotations** → **Payment Milestones** (hasMany)
- **Users** → **Roles** (belongsToMany)
- **Roles** → **Permissions** (belongsToMany)

---

## API Documentation

### Authentication Endpoints

#### POST /api/v1/auth/login
**Description**: User authentication
**Request Body**:
```json
{
  "email": "user@example.com",
  "password": "password123"
}
```
**Response**:
```json
{
  "message": "Login successful",
  "access_token": "token_string",
  "token_type": "Bearer",
  "user": {
    "id": 1,
    "name": "User Name",
    "email": "user@example.com",
    "roles": [...],
    "permissions": [...]
  }
}
```

#### POST /api/v1/auth/logout
**Description**: User logout
**Headers**: Authorization: Bearer {token}
**Response**:
```json
{
  "message": "Logout successful"
}
```

#### GET /api/v1/user
**Description**: Get authenticated user profile
**Headers**: Authorization: Bearer {token}
**Response**:
```json
{
  "user": {
    "id": 1,
    "name": "User Name",
    "email": "user@example.com",
    "profile": {...},
    "roles": [...],
    "permissions": [...]
  }
}
```

### Project Management Endpoints

#### GET /api/v1/projects
**Description**: List all projects with pagination and filters
**Headers**: Authorization: Bearer {token}
**Query Parameters**:
- `page`: Page number (default: 1)
- `per_page`: Items per page (default: 15)
- `status`: Filter by status
- `client_id`: Filter by client
- `search`: Search in project name/description

**Response**:
```json
{
  "data": [...],
  "pagination": {
    "current_page": 1,
    "last_page": 5,
    "per_page": 15,
    "total": 75
  }
}
```

#### POST /api/v1/projects
**Description**: Create new project
**Headers**: Authorization: Bearer {token}
**Request Body**:
```json
{
  "name": "Project Name",
  "description": "Project Description",
  "client_id": 1,
  "start_date": "2025-01-01",
  "end_date": "2025-12-31",
  "status": "planning",
  "total_budget": 100000.00
}
```

#### GET /api/v1/projects/{id}
**Description**: Get project details with stages and activities
**Headers**: Authorization: Bearer {token}

#### PUT /api/v1/projects/{id}
**Description**: Update project
**Headers**: Authorization: Bearer {token}

#### DELETE /api/v1/projects/{id}
**Description**: Delete project (soft delete)
**Headers**: Authorization: Bearer {token}

### Client Management Endpoints

#### GET /api/v1/clients
**Description**: List all clients
**Headers**: Authorization: Bearer {token}
**Query Parameters**:
- `type`: Filter by client type (individual/corporate)
- `status`: Filter by status
- `search`: Search in name/email/company

#### POST /api/v1/clients
**Description**: Create new client
**Headers**: Authorization: Bearer {token}
**Request Body**:
```json
{
  "type": "corporate",
  "name": "Client Name",
  "company_name": "Company Name",
  "email": "client@example.com",
  "phone": "+1234567890",
  "address": "Full Address",
  "city": "City",
  "state": "State",
  "pincode": "12345"
}
```

#### GET /api/v1/clients/options
**Description**: Get client options for dropdowns
**Headers**: Authorization: Bearer {token}
**Response**:
```json
{
  "data": [
    {
      "id": 1,
      "name": "Client Name",
      "label": "Client Name (client@example.com)"
    }
  ]
}
```

### Financial Management Endpoints

#### GET /api/v1/projects/{id}/quotations
**Description**: Get project quotations
**Headers**: Authorization: Bearer {token}

#### POST /api/v1/projects/{id}/quotations
**Description**: Create quotation
**Headers**: Authorization: Bearer {token}
**Request Body**:
```json
{
  "quotation_number": "QTN-001",
  "valid_until": "2025-12-31",
  "subtotal": 10000.00,
  "tax_rate": 18.00,
  "terms_conditions": "Terms text",
  "notes": "Additional notes"
}
```

#### GET /api/v1/projects/{id}/expenses
**Description**: Get project expenses
**Headers**: Authorization: Bearer {token}
**Query Parameters**:
- `category`: Filter by expense category
- `status`: Filter by approval status
- `date_from`: Filter from date
- `date_to`: Filter to date

#### POST /api/v1/projects/{id}/expenses
**Description**: Create expense
**Headers**: Authorization: Bearer {token}
**Request Body**:
```json
{
  "title": "Expense Title",
  "description": "Expense description",
  "category": "materials",
  "amount": 500.00,
  "expense_date": "2025-01-15",
  "receipt_path": "path/to/receipt.jpg"
}
```

#### POST /api/v1/expenses/{id}/approve
**Description**: Approve expense
**Headers**: Authorization: Bearer {token}

#### POST /api/v1/expenses/{id}/reject
**Description**: Reject expense
**Headers**: Authorization: Bearer {token}
**Request Body**:
```json
{
  "rejection_reason": "Reason for rejection"
}
```

---

## Frontend Architecture

### Component Structure
```
src/app/
├── components/              # Shared Components
│   ├── data-table/         # Reusable data table
│   ├── form-fields/        # Custom form components
│   └── charts/              # Chart components
├── services/               # Angular Services
│   ├── api.service.ts      # API communication
│   ├── auth.service.ts     # Authentication
│   └── notification.service.ts
├── guards/                 # Route Guards
│   ├── auth.guard.ts       # Authentication guard
│   └── role.guard.ts      # Role-based access
├── interceptors/           # HTTP Interceptors
│   └── auth.interceptor.ts # Token injection
├── models/                 # TypeScript Interfaces
│   ├── project.interface.ts
│   ├── client.interface.ts
│   └── user.interface.ts
└── modules/                # Feature Modules
    ├── projects/           # Project Management
    ├── clients/            # Client Management
    ├── financial/          # Financial Management
    └── dashboard/          # Dashboard
```

### Service Architecture

#### API Service
```typescript
@Injectable({
  providedIn: 'root'
})
export class ApiService {
  private apiUrl = environment.apiUrl;
  
  // Authentication methods
  login(email: string, password: string): Observable<AuthResponse>
  logout(): Observable<any>
  getUserProfile(): Observable<any>
  
  // Project methods
  getProjects(params?: any): Observable<ApiResponse<Project[]>>
  createProject(project: ProjectFormData): Observable<ApiResponse<Project>>
  updateProject(id: number, project: ProjectFormData): Observable<ApiResponse<Project>>
  deleteProject(id: number): Observable<ApiResponse<any>>
  
  // Client methods
  getClients(params?: any): Observable<ApiResponse<Client[]>>
  createClient(client: ClientFormData): Observable<ApiResponse<Client>>
  updateClient(id: number, client: ClientFormData): Observable<ApiResponse<Client>>
  deleteClient(id: number): Observable<ApiResponse<any>>
}
```

#### Authentication Service
```typescript
@Injectable({
  providedIn: 'root'
})
export class AuthService {
  private currentUserSubject = new BehaviorSubject<any>(null);
  public currentUser$ = this.currentUserSubject.asObservable();
  
  login(email: string, password: string): Observable<AuthResponse>
  logout(): Observable<any>
  getToken(): string | null
  isAuthenticated(): boolean
  hasRole(role: string): boolean
  hasPermission(permission: string): boolean
}
```

### Material Design Implementation

#### Component Usage
```typescript
// Data Table
<mat-table [dataSource]="dataSource">
  <ng-container matColumnDef="name">
    <th mat-header-cell *matHeaderCellDef>Name</th>
    <td mat-cell *matCellDef="let element">{{element.name}}</td>
  </ng-container>
</mat-table>

// Form Fields
<mat-form-field appearance="outline">
  <mat-label>Project Name</mat-label>
  <input matInput formControlName="name">
  <mat-error *ngIf="isFieldInvalid('name')">{{getFieldError('name')}}</mat-error>
</mat-form-field>

// Cards
<mat-card>
  <mat-card-header>
    <mat-card-title>Project Details</mat-card-title>
  </mat-card-header>
  <mat-card-content>
    <!-- Content -->
  </mat-card-content>
</mat-card>
```

---

## Authentication & Security

### Laravel Sanctum Implementation
```php
// config/sanctum.php
'stateful' => explode(',', env('SANCTUM_STATEFUL_DOMAINS', 
    'localhost,localhost:4200,127.0.0.1,127.0.0.1:8000'
)),

'guard' => ['web'],
'expiration' => null,
```

### Role-Based Access Control
```php
// Middleware for permission checking
class CheckPermission
{
    public function handle($request, Closure $next, $permission)
    {
        if (!$request->user()->hasPermission($permission)) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }
        return $next($request);
    }
}
```

### Frontend Route Protection
```typescript
// Auth Guard
@Injectable()
export class AuthGuard implements CanActivate {
  constructor(private authService: AuthService, private router: Router) {}
  
  canActivate(): boolean {
    if (this.authService.isAuthenticated()) {
      return true;
    }
    this.router.navigate(['/login']);
    return false;
  }
}

// Role Guard
@Injectable()
export class RoleGuard implements CanActivate {
  canActivate(route: ActivatedRouteSnapshot): boolean {
    const requiredRoles = route.data['roles'] as string[];
    return this.authService.hasAnyRole(requiredRoles);
  }
}
```

---

## Deployment Guide

### Backend Deployment (Laravel)

#### Environment Setup
```bash
# Install dependencies
composer install --optimize-autoloader --no-dev

# Generate application key
php artisan key:generate

# Run migrations
php artisan migrate --force

# Seed database
php artisan db:seed --force

# Cache configuration
php artisan config:cache
php artisan route:cache
php artisan view:cache
```

#### Web Server Configuration (Nginx)
```nginx
server {
    listen 80;
    server_name your-domain.com;
    root /path/to/construction-erp/backend/public;
    
    add_header X-Frame-Options "SAMEORIGIN";
    add_header X-Content-Type-Options "nosniff";
    
    index index.php;
    
    charset utf-8;
    
    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }
    
    location = /favicon.ico { access_log off; log_not_found off; }
    location = /robots.txt  { access_log off; log_not_found off; }
    
    error_page 404 /index.php;
    
    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php/php8.2-fpm.sock;
        fastcgi_param SCRIPT_FILENAME $realpath_root$fastcgi_script_name;
        include fastcgi_params;
    }
    
    location ~ /\.(?!well-known).* {
        deny all;
    }
}
```

### Frontend Deployment (Angular)

#### Build for Production
```bash
# Install dependencies
npm install

# Build for production
ng build --configuration production

# The build artifacts will be stored in the `dist/` directory
```

#### Web Server Configuration
```nginx
server {
    listen 80;
    server_name your-frontend-domain.com;
    root /path/to/construction-erp/frontend/dist;
    index index.html;
    
    location / {
        try_files $uri $uri/ /index.html;
    }
    
    location ~* \.(js|css|png|jpg|jpeg|gif|ico|svg)$ {
        expires 1y;
        add_header Cache-Control "public, immutable";
    }
}
```

---

## Development Setup

### Prerequisites
- **PHP**: 8.2 or higher
- **Composer**: Latest version
- **Node.js**: 18.x or higher
- **MySQL**: 8.0 or higher
- **Git**: Latest version

### Backend Setup
```bash
# Clone repository
git clone <repository-url>
cd ConstructionERP/backend

# Install dependencies
composer install

# Copy environment file
cp .env.example .env

# Generate application key
php artisan key:generate

# Configure database in .env
DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=construction_erp
DB_USERNAME=root
DB_PASSWORD=password

# Run migrations
php artisan migrate

# Seed database
php artisan db:seed

# Start development server
php artisan serve
```

### Frontend Setup
```bash
# Navigate to frontend directory
cd ConstructionERP/frontend

# Install dependencies
npm install

# Configure environment
cp src/environments/environment.example.ts src/environments/environment.ts

# Update API URL in environment.ts
export const environment = {
  production: false,
  apiUrl: 'http://localhost:8000/api/v1'
};

# Start development server
ng serve
```

### Database Seeding
```bash
# Seed with sample data
php artisan db:seed --class=RolePermissionSeeder
php artisan db:seed --class=ClientSeeder

# Create additional test data
php artisan tinker
>>> App\Models\Project::factory(10)->create();
>>> App\Models\Client::factory(20)->create();
```

---

## Testing Guidelines

### Backend Testing (PHPUnit)
```bash
# Run all tests
php artisan test

# Run specific test
php artisan test --filter=ProjectTest

# Run with coverage
php artisan test --coverage
```

#### Test Structure
```php
// tests/Feature/ProjectTest.php
class ProjectTest extends TestCase
{
    use RefreshDatabase;
    
    public function test_can_create_project()
    {
        $user = User::factory()->create();
        $client = Client::factory()->create();
        
        $response = $this->actingAs($user)
            ->postJson('/api/v1/projects', [
                'name' => 'Test Project',
                'client_id' => $client->id,
                'total_budget' => 100000
            ]);
            
        $response->assertStatus(201)
            ->assertJsonStructure([
                'data' => ['id', 'name', 'client_id']
            ]);
    }
}
```

### Frontend Testing (Jasmine/Karma)
```bash
# Run unit tests
ng test

# Run e2e tests
ng e2e

# Run tests with coverage
ng test --code-coverage
```

#### Component Testing
```typescript
// src/app/projects/project-list/project-list.component.spec.ts
describe('ProjectListComponent', () => {
  let component: ProjectListComponent;
  let fixture: ComponentFixture<ProjectListComponent>;
  
  beforeEach(() => {
    TestBed.configureTestingModule({
      declarations: [ProjectListComponent],
      imports: [MaterialModule, HttpClientTestingModule]
    });
    fixture = TestBed.createComponent(ProjectListComponent);
    component = fixture.componentInstance;
  });
  
  it('should create', () => {
    expect(component).toBeTruthy();
  });
});
```

---

## Performance Optimization

### Backend Optimization

#### Database Optimization
```php
// Eager loading relationships
$projects = Project::with(['client', 'stages.activities'])->get();

// Query optimization
$projects = Project::select(['id', 'name', 'status'])
    ->where('status', 'active')
    ->paginate(15);

// Database indexing
Schema::table('projects', function (Blueprint $table) {
    $table->index(['status', 'created_at']);
    $table->index('client_id');
});
```

#### Caching Implementation
```php
// Cache frequently accessed data
$projects = Cache::remember('projects.active', 3600, function () {
    return Project::active()->with('client')->get();
});

// Cache configuration
php artisan config:cache
php artisan route:cache
php artisan view:cache
```

### Frontend Optimization

#### Lazy Loading
```typescript
// Lazy load modules
const routes: Routes = [
  {
    path: 'projects',
    loadChildren: () => import('./projects/projects.module').then(m => m.ProjectsModule)
  }
];
```

#### OnPush Change Detection
```typescript
@Component({
  selector: 'app-project-list',
  changeDetection: ChangeDetectionStrategy.OnPush
})
export class ProjectListComponent {
  // Component implementation
}
```

#### Virtual Scrolling
```html
<cdk-virtual-scroll-viewport itemSize="50" class="viewport">
  <div *cdkVirtualFor="let project of projects" class="item">
    {{project.name}}
  </div>
</cdk-virtual-scroll-viewport>
```

---

## Troubleshooting

### Common Backend Issues

#### Database Connection Issues
```bash
# Check database connection
php artisan tinker
>>> DB::connection()->getPdo();

# Reset database
php artisan migrate:fresh --seed
```

#### Permission Issues
```bash
# Set proper permissions
chmod -R 755 storage bootstrap/cache
chown -R www-data:www-data storage bootstrap/cache
```

#### Memory Issues
```bash
# Increase PHP memory limit
php -d memory_limit=512M artisan migrate
```

### Common Frontend Issues

#### Build Issues
```bash
# Clear Angular cache
rm -rf node_modules package-lock.json
npm install

# Clear build cache
ng build --delete-output-path
```

#### CORS Issues
```php
// config/cors.php
'allowed_origins' => ['http://localhost:4200'],
'allowed_methods' => ['*'],
'allowed_headers' => ['*'],
'supports_credentials' => true,
```

#### Authentication Issues
```typescript
// Check token validity
if (this.authService.isAuthenticated()) {
  // Token exists and valid
} else {
  // Redirect to login
  this.router.navigate(['/login']);
}
```

### Performance Monitoring

#### Backend Monitoring
```php
// Log slow queries
DB::listen(function ($query) {
    if ($query->time > 1000) {
        Log::warning('Slow query detected', [
            'sql' => $query->sql,
            'time' => $query->time
        ]);
    }
});
```

#### Frontend Monitoring
```typescript
// Performance monitoring
import { NgZone } from '@angular/core';

constructor(private ngZone: NgZone) {
  this.ngZone.runOutsideAngular(() => {
    // Performance monitoring code
  });
}
```

---

## Conclusion

This technical documentation provides comprehensive guidance for developers and system administrators working with the Construction ERP System. It covers all aspects from system architecture to deployment and troubleshooting.

**Last Updated**: September 2025
**Version**: 1.0
**System**: Construction ERP v1.0






