# Petty Cash Management System - Architecture Document

## Overview
This document outlines the architecture for implementing a comprehensive Petty Cash management system within the existing Construction ERP. The system will track petty cash allocations to staff members and manage expenses paid through petty cash.

## Current System Analysis

### Existing Infrastructure
- **Expense Model**: `app/Models/Expense.php` with basic fields
- **Expense Controller**: `app/Http/Controllers/ExpenseController.php` using Bill model
- **Database**: Basic expenses table with employee_id field
- **UI**: Simple expense form with staff selection and payment mode

### Identified Gaps
1. No dedicated petty cash payment mode
2. No petty cash allocation tracking
3. No balance management for staff petty cash
4. No petty cash specific validation rules

## Proposed Architecture

### 1. Database Schema Enhancement

#### A. Petty Cash Allocations Table
```sql
CREATE TABLE petty_cash_allocations (
    id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
    employee_id BIGINT UNSIGNED NOT NULL,
    allocated_amount DECIMAL(10,2) NOT NULL,
    current_balance DECIMAL(10,2) NOT NULL,
    allocation_date DATE NOT NULL,
    allocated_by BIGINT UNSIGNED NOT NULL,
    purpose TEXT,
    status ENUM('active', 'closed', 'suspended') DEFAULT 'active',
    created_at TIMESTAMP,
    updated_at TIMESTAMP,
    FOREIGN KEY (employee_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (allocated_by) REFERENCES users(id)
);
```

#### B. Petty Cash Transactions Table
```sql
CREATE TABLE petty_cash_transactions (
    id BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
    allocation_id BIGINT UNSIGNED NOT NULL,
    employee_id BIGINT UNSIGNED NOT NULL,
    expense_id BIGINT UNSIGNED NULL,
    transaction_type ENUM('allocation', 'expense', 'return', 'adjustment') NOT NULL,
    amount DECIMAL(10,2) NOT NULL,
    balance_after DECIMAL(10,2) NOT NULL,
    description TEXT,
    receipt_number VARCHAR(100),
    transaction_date DATE NOT NULL,
    project_id BIGINT UNSIGNED NULL,
    created_by BIGINT UNSIGNED NOT NULL,
    created_at TIMESTAMP,
    updated_at TIMESTAMP,
    FOREIGN KEY (allocation_id) REFERENCES petty_cash_allocations(id),
    FOREIGN KEY (employee_id) REFERENCES users(id),
    FOREIGN KEY (expense_id) REFERENCES expenses(id),
    FOREIGN KEY (project_id) REFERENCES projects(id),
    FOREIGN KEY (created_by) REFERENCES users(id)
);
```

#### C. Expense Table Enhancement
```sql
-- Add new fields to existing expenses table
ALTER TABLE expenses ADD COLUMN payment_mode ENUM('bank', 'cash', 'petty_cash', 'cheque') DEFAULT 'bank';
ALTER TABLE expenses ADD COLUMN petty_cash_allocation_id BIGINT UNSIGNED NULL;
ALTER TABLE expenses ADD COLUMN staff_id BIGINT UNSIGNED NULL;

-- Add foreign key constraints
ALTER TABLE expenses ADD FOREIGN KEY (petty_cash_allocation_id) REFERENCES petty_cash_allocations(id);
ALTER TABLE expenses ADD FOREIGN KEY (staff_id) REFERENCES users(id);
```

### 2. Model Architecture

#### A. PettyCashAllocation Model
```php
class PettyCashAllocation extends Model
{
    protected $fillable = [
        'employee_id', 'allocated_amount', 'current_balance', 
        'allocation_date', 'allocated_by', 'purpose', 'status'
    ];

    // Relationships
    public function employee() { return $this->belongsTo(User::class, 'employee_id'); }
    public function allocatedBy() { return $this->belongsTo(User::class, 'allocated_by'); }
    public function transactions() { return $this->hasMany(PettyCashTransaction::class, 'allocation_id'); }
    public function expenses() { return $this->hasMany(Expense::class, 'petty_cash_allocation_id'); }

    // Business Logic Methods
    public function deductAmount($amount) { /* Update balance */ }
    public function addAmount($amount) { /* Add to balance */ }
    public function getAvailableBalance() { /* Return current balance */ }
}
```

#### B. PettyCashTransaction Model
```php
class PettyCashTransaction extends Model
{
    protected $fillable = [
        'allocation_id', 'employee_id', 'expense_id', 'transaction_type',
        'amount', 'balance_after', 'description', 'receipt_number',
        'transaction_date', 'project_id', 'created_by'
    ];

    // Relationships
    public function allocation() { return $this->belongsTo(PettyCashAllocation::class); }
    public function employee() { return $this->belongsTo(User::class, 'employee_id'); }
    public function expense() { return $this->belongsTo(Expense::class); }
    public function project() { return $this->belongsTo(Project::class); }
    public function createdBy() { return $this->belongsTo(User::class, 'created_by'); }
}
```

#### C. Enhanced Expense Model
```php
// Add to existing Expense model
public function pettyCashAllocation() { 
    return $this->belongsTo(PettyCashAllocation::class, 'petty_cash_allocation_id'); 
}

public function staffMember() { 
    return $this->belongsTo(User::class, 'staff_id'); 
}

public function pettyCashTransaction() { 
    return $this->hasOne(PettyCashTransaction::class, 'expense_id'); 
}
```

### 3. Controller Architecture

#### A. PettyCashController
```php
class PettyCashController extends Controller
{
    // Allocation Management
    public function allocateIndex() { /* List all allocations */ }
    public function allocateCreate() { /* Show allocation form */ }
    public function allocateStore() { /* Create new allocation */ }
    public function allocateShow($id) { /* Show allocation details */ }
    public function allocateEdit($id) { /* Edit allocation */ }
    public function allocateUpdate($id) { /* Update allocation */ }
    
    // Balance Management
    public function adjustBalance($id) { /* Adjust allocation balance */ }
    public function closeAllocation($id) { /* Close allocation */ }
    
    // Reporting
    public function allocationReport() { /* Allocation summary report */ }
    public function transactionReport() { /* Transaction history report */ }
    public function balanceReport() { /* Current balance report */ }
}
```

#### B. Enhanced ExpenseController
```php
// Add to existing ExpenseController
public function getPettyCashAllocations(Request $request) {
    // Return active petty cash allocations for selected employee
    $employee_id = $request->employee_id;
    return PettyCashAllocation::where('employee_id', $employee_id)
        ->where('status', 'active')
        ->where('current_balance', '>', 0)
        ->get();
}

// Modify store method to handle petty cash expenses
public function store(Request $request) {
    // Existing validation + petty cash validation
    if ($request->payment_mode === 'petty_cash') {
        $this->validatePettyCashExpense($request);
    }
    
    // Create expense and petty cash transaction
    DB::transaction(function() use ($request) {
        $expense = $this->createExpense($request);
        
        if ($request->payment_mode === 'petty_cash') {
            $this->processPettyCashExpense($expense, $request);
        }
    });
}

private function validatePettyCashExpense($request) {
    // Validate staff selection is mandatory
    // Validate petty cash allocation exists and has sufficient balance
    // Validate amount doesn't exceed available balance
}

private function processPettyCashExpense($expense, $request) {
    // Deduct amount from petty cash allocation
    // Create petty cash transaction record
    // Update allocation balance
}
```

### 4. UI/UX Enhancements

#### A. Expense Form Modifications
1. **Payment Mode Selection**: Add "Petty Cash" option to payment mode dropdown
2. **Staff Selection**: Make staff selection mandatory when Petty Cash is selected
3. **Allocation Selection**: Show available petty cash allocations for selected staff
4. **Balance Display**: Show current balance for selected allocation
5. **Validation**: Real-time validation for sufficient balance

#### B. New UI Components
1. **Petty Cash Allocation Management**
   - List all allocations with current balances
   - Create/Edit allocation forms
   - Allocation details view with transaction history

2. **Petty Cash Dashboard**
   - Summary cards: Total Allocated, Total Spent, Available Balance
   - Recent transactions list
   - Low balance alerts

3. **Reporting Interface**
   - Allocation summary reports
   - Staff-wise petty cash usage
   - Project-wise petty cash expenses
   - Monthly/Quarterly petty cash reports

### 5. Business Rules & Validation

#### A. Allocation Rules
- Only authorized users can create allocations
- Maximum allocation limit per employee (configurable)
- Allocation must have a valid purpose
- Cannot allocate to inactive employees

#### B. Expense Rules
- Staff selection mandatory for petty cash expenses
- Cannot exceed available allocation balance
- Must have valid receipt/documentation
- Auto-deduction from allocation balance

#### C. Balance Management
- Real-time balance updates
- Automatic allocation closure when balance reaches zero
- Balance adjustment with proper authorization
- Return unused amounts to company account

### 6. Security & Permissions

#### A. Role-Based Access
- **Admin**: Full access to all petty cash functions
- **Manager**: Can allocate and view reports
- **Staff**: Can only view their own allocations and create expenses
- **Accountant**: Can view all transactions and generate reports

#### B. Audit Trail
- All petty cash transactions logged
- Balance changes tracked with timestamps
- User actions recorded for accountability

### 7. Integration Points

#### A. Existing Modules
- **Project Management**: Link petty cash expenses to projects
- **Accounting**: Auto-generate journal entries
- **Reporting**: Include in financial reports
- **User Management**: Employee selection and permissions

#### B. API Endpoints
- GET /api/petty-cash/allocations
- POST /api/petty-cash/allocate
- GET /api/petty-cash/balance/{employee_id}
- POST /api/petty-cash/expense
- GET /api/petty-cash/transactions

### 8. Implementation Phases

#### Phase 1: Database & Models (Week 1)
- Create migration files
- Implement models with relationships
- Set up basic validation rules

#### Phase 2: Core Functionality (Week 2)
- Implement PettyCashController
- Enhance ExpenseController
- Create allocation management

#### Phase 3: UI Development (Week 3)
- Modify expense form
- Create allocation management interface
- Implement dashboard components

#### Phase 4: Reporting & Testing (Week 4)
- Develop reporting features
- Comprehensive testing
- Documentation and deployment

### 9. Success Metrics

#### A. Functional Metrics
- Successful petty cash allocation creation
- Accurate balance tracking
- Proper expense validation
- Complete audit trail

#### B. User Experience Metrics
- Reduced time for expense entry
- Improved accuracy in petty cash tracking
- Better visibility into cash flow
- Enhanced financial control

### 10. Risk Mitigation

#### A. Technical Risks
- **Data Integrity**: Use database transactions for all operations
- **Performance**: Implement proper indexing and caching
- **Security**: Validate all inputs and implement proper authorization

#### B. Business Risks
- **Over-allocation**: Implement maximum limits and approval workflows
- **Misuse**: Comprehensive audit trails and regular reconciliation
- **Balance Discrepancies**: Automated balance validation and alerts

## Conclusion

This architecture provides a comprehensive solution for petty cash management while integrating seamlessly with the existing Construction ERP system. The phased implementation approach ensures minimal disruption to current operations while delivering immediate value to users.