# Construction Module Adaptation Plan
## Modifying Existing Project Module for Construction ERP

---

## Current System Analysis

### Existing Structure
```
Current: Project → Tasks (TaskStage is global, not project-specific)
Target:  Project → Stages → Activities (Construction-specific hierarchy)
```

### Current Models & Tables
1. **Project** (`projects` table)
   - Fields: id, project_name, start_date, end_date, client_id, budget, status, etc.
   - Relationships: hasMany tasks, hasMany milestones, belongsTo client

2. **TaskStage** (`task_stages` table) - **GLOBAL for all projects**
   - Fields: id, name, color, order, created_by
   - Used as workflow stages (like Kanban: To Do, In Progress, Done)
   - NOT project-specific

3. **ProjectTask** (`project_tasks` table)
   - Fields: id, project_id, stage_id, name, estimated_hrs, assign_to, priority, etc.
   - Relationships: belongsTo project, belongsTo stage

4. **Projectstages** (`projectstages` table)
   - Seems to be a different model for project stages
   - Need to investigate relationship

### Key Observations
- `TaskStage` is a GLOBAL workflow stage system (not project-specific construction stages)
- Need to create NEW tables for construction stages/activities
- Can't directly use existing `TaskStage` for construction phases

---

## Adaptation Strategy

### Option A: Create Separate Construction Tables (RECOMMENDED)
Keep existing project/task system intact and create parallel construction-specific tables.

**New Tables:**
```sql
construction_stages:
  - id
  - project_id (FK to projects)
  - name (Foundation, Framing, Roofing, etc.)
  - description
  - sequence_order
  - budget_allocated
  - start_date
  - end_date
  - status (pending/in_progress/completed)
  - created_by

construction_activities:
  - id
  - construction_stage_id (FK to construction_stages)
  - project_id (FK to projects - for easier queries)
  - name
  - description
  - estimated_cost
  - actual_cost
  - labor_hours
  - material_cost
  - equipment_cost
  - status (pending/in_progress/completed/rework)
  - progress_percentage
  - assigned_to
  - start_date
  - end_date
  - created_by
```

**New Models:**
- `ConstructionStage.php`
- `ConstructionActivity.php`

**Advantages:**
✅ No breaking changes to existing system
✅ Clear separation of concerns
✅ Existing Project/Task features remain available
✅ Can run both systems in parallel
✅ Easier to maintain

**Disadvantages:**
❌ Some data duplication
❌ Need to maintain both systems

---

### Option B: Extend Existing Tables (NOT RECOMMENDED)
Add fields to existing tables and use type flags.

**Problems:**
❌ Would break existing functionality
❌ Complex conditional logic everywhere
❌ Difficult to maintain
❌ Risk of data corruption

---

## Implementation Plan (Option A)

### Phase 1: Database Setup

#### 1.1 Create Migrations

**File:** `database/migrations/2025_10_09_create_construction_stages_table.php`
```php
<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

class CreateConstructionStagesTable extends Migration
{
    public function up()
    {
        Schema::create('construction_stages', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('project_id');
            $table->string('name');
            $table->text('description')->nullable();
            $table->integer('sequence_order')->default(0);
            $table->decimal('budget_allocated', 15, 2)->default(0);
            $table->date('start_date')->nullable();
            $table->date('end_date')->nullable();
            $table->enum('status', ['pending', 'in_progress', 'completed', 'on_hold'])->default('pending');
            $table->unsignedBigInteger('created_by');
            $table->timestamps();

            $table->foreign('project_id')->references('id')->on('projects')->onDelete('cascade');
            $table->foreign('created_by')->references('id')->on('users')->onDelete('cascade');
        });
    }

    public function down()
    {
        Schema::dropIfExists('construction_stages');
    }
}
```

**File:** `database/migrations/2025_10_09_create_construction_activities_table.php`
```php
<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

class CreateConstructionActivitiesTable extends Migration
{
    public function up()
    {
        Schema::create('construction_activities', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('construction_stage_id');
            $table->unsignedBigInteger('project_id'); // for easier queries
            $table->string('name');
            $table->text('description')->nullable();
            $table->decimal('estimated_cost', 15, 2)->default(0);
            $table->decimal('actual_cost', 15, 2)->default(0);
            $table->decimal('labor_hours', 8, 2)->default(0);
            $table->decimal('material_cost', 15, 2)->default(0);
            $table->decimal('equipment_cost', 15, 2)->default(0);
            $table->enum('status', ['pending', 'in_progress', 'completed', 'rework'])->default('pending');
            $table->integer('progress_percentage')->default(0);
            $table->text('assigned_to')->nullable(); // comma-separated user IDs
            $table->date('start_date')->nullable();
            $table->date('end_date')->nullable();
            $table->unsignedBigInteger('created_by');
            $table->timestamps();

            $table->foreign('construction_stage_id')->references('id')->on('construction_stages')->onDelete('cascade');
            $table->foreign('project_id')->references('id')->on('projects')->onDelete('cascade');
            $table->foreign('created_by')->references('id')->on('users')->onDelete('cascade');
        });
    }

    public function down()
    {
        Schema::dropIfExists('construction_activities');
    }
}
```

#### 1.2 Create Seeder for Default Construction Stages

**File:** `database/seeders/ConstructionStageTemplateSeeder.php`
```php
<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;

class ConstructionStageTemplateSeeder extends Seeder
{
    public function run()
    {
        $stages = [
            ['name' => 'Site Preparation', 'sequence' => 1],
            ['name' => 'Foundation & Excavation', 'sequence' => 2],
            ['name' => 'Structural Framework', 'sequence' => 3],
            ['name' => 'Roofing', 'sequence' => 4],
            ['name' => 'Plumbing Installation', 'sequence' => 5],
            ['name' => 'Electrical Installation', 'sequence' => 6],
            ['name' => 'Masonry & Plastering', 'sequence' => 7],
            ['name' => 'Flooring', 'sequence' => 8],
            ['name' => 'Painting & Finishing', 'sequence' => 9],
            ['name' => 'Final Inspection', 'sequence' => 10],
        ];

        DB::table('construction_stage_templates')->insert($stages);
    }
}
```

---

### Phase 2: Create Models

#### 2.1 ConstructionStage Model

**File:** `app/Models/ConstructionStage.php`
```php
<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class ConstructionStage extends Model
{
    protected $fillable = [
        'project_id',
        'name',
        'description',
        'sequence_order',
        'budget_allocated',
        'start_date',
        'end_date',
        'status',
        'created_by',
    ];

    protected $casts = [
        'budget_allocated' => 'decimal:2',
        'start_date' => 'date',
        'end_date' => 'date',
    ];

    public static $statuses = [
        'pending' => 'Pending',
        'in_progress' => 'In Progress',
        'completed' => 'Completed',
        'on_hold' => 'On Hold',
    ];

    public static $status_colors = [
        'pending' => 'secondary',
        'in_progress' => 'primary',
        'completed' => 'success',
        'on_hold' => 'warning',
    ];

    // Relationships
    public function project()
    {
        return $this->belongsTo(Project::class);
    }

    public function activities()
    {
        return $this->hasMany(ConstructionActivity::class)->orderBy('id', 'desc');
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    // Helper Methods
    public function getBudgetSpent()
    {
        return $this->activities()->sum('actual_cost');
    }

    public function getBudgetRemaining()
    {
        return $this->budget_allocated - $this->getBudgetSpent();
    }

    public function getProgressPercentage()
    {
        $total = $this->activities()->count();
        if ($total == 0) return 0;

        $completed = $this->activities()->where('status', 'completed')->count();
        return round(($completed / $total) * 100);
    }
}
```

#### 2.2 ConstructionActivity Model

**File:** `app/Models/ConstructionActivity.php`
```php
<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class ConstructionActivity extends Model
{
    protected $fillable = [
        'construction_stage_id',
        'project_id',
        'name',
        'description',
        'estimated_cost',
        'actual_cost',
        'labor_hours',
        'material_cost',
        'equipment_cost',
        'status',
        'progress_percentage',
        'assigned_to',
        'start_date',
        'end_date',
        'created_by',
    ];

    protected $casts = [
        'estimated_cost' => 'decimal:2',
        'actual_cost' => 'decimal:2',
        'labor_hours' => 'decimal:2',
        'material_cost' => 'decimal:2',
        'equipment_cost' => 'decimal:2',
        'start_date' => 'date',
        'end_date' => 'date',
    ];

    public static $statuses = [
        'pending' => 'Pending',
        'in_progress' => 'In Progress',
        'completed' => 'Completed',
        'rework' => 'Rework Required',
    ];

    public static $status_colors = [
        'pending' => 'secondary',
        'in_progress' => 'primary',
        'completed' => 'success',
        'rework' => 'danger',
    ];

    // Relationships
    public function stage()
    {
        return $this->belongsTo(ConstructionStage::class, 'construction_stage_id');
    }

    public function project()
    {
        return $this->belongsTo(Project::class);
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function assignedUsers()
    {
        if (empty($this->assigned_to)) {
            return collect([]);
        }
        return User::whereIn('id', explode(',', $this->assigned_to))->get();
    }

    // Helper Methods
    public function getCostVariance()
    {
        return $this->estimated_cost - $this->actual_cost;
    }

    public function isOverBudget()
    {
        return $this->actual_cost > $this->estimated_cost;
    }
}
```

#### 2.3 Update Project Model

**File:** `app/Models/Project.php` (add these methods)
```php
// Add to existing Project model

public function constructionStages()
{
    return $this->hasMany(ConstructionStage::class)->orderBy('sequence_order');
}

public function constructionActivities()
{
    return $this->hasMany(ConstructionActivity::class);
}

public function getTotalConstructionBudget()
{
    return $this->constructionStages()->sum('budget_allocated');
}

public function getTotalConstructionSpent()
{
    return $this->constructionActivities()->sum('actual_cost');
}

public function getConstructionProgress()
{
    $totalStages = $this->constructionStages()->count();
    if ($totalStages == 0) return 0;

    $completedStages = $this->constructionStages()->where('status', 'completed')->count();
    return round(($completedStages / $totalStages) * 100);
}
```

---

### Phase 3: Create Controllers

#### 3.1 ConstructionStageController

**File:** `app/Http/Controllers/ConstructionStageController.php`
```php
<?php

namespace App\Http\Controllers;

use App\Models\Project;
use App\Models\ConstructionStage;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class ConstructionStageController extends Controller
{
    public function index($project_id)
    {
        if (Auth::user()->can('manage construction')) {
            $project = Project::findOrFail($project_id);
            $stages = $project->constructionStages;

            return view('construction.stages.index', compact('project', 'stages'));
        }
        return redirect()->back()->with('error', __('Permission Denied.'));
    }

    public function create($project_id)
    {
        if (Auth::user()->can('create construction stage')) {
            $project = Project::findOrFail($project_id);
            return view('construction.stages.create', compact('project'));
        }
        return redirect()->back()->with('error', __('Permission Denied.'));
    }

    public function store(Request $request, $project_id)
    {
        if (Auth::user()->can('create construction stage')) {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'budget_allocated' => 'required|numeric|min:0',
            ]);

            if ($validator->fails()) {
                return redirect()->back()->with('error', $validator->errors()->first());
            }

            $stage = new ConstructionStage();
            $stage->project_id = $project_id;
            $stage->name = $request->name;
            $stage->description = $request->description;
            $stage->budget_allocated = $request->budget_allocated;
            $stage->start_date = $request->start_date;
            $stage->end_date = $request->end_date;
            $stage->status = $request->status ?? 'pending';
            $stage->sequence_order = ConstructionStage::where('project_id', $project_id)->max('sequence_order') + 1;
            $stage->created_by = Auth::user()->id;
            $stage->save();

            return redirect()->route('construction.stages.index', $project_id)
                           ->with('success', __('Construction stage created successfully.'));
        }
        return redirect()->back()->with('error', __('Permission Denied.'));
    }

    public function edit($id)
    {
        if (Auth::user()->can('edit construction stage')) {
            $stage = ConstructionStage::findOrFail($id);
            return view('construction.stages.edit', compact('stage'));
        }
        return redirect()->back()->with('error', __('Permission Denied.'));
    }

    public function update(Request $request, $id)
    {
        if (Auth::user()->can('edit construction stage')) {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'budget_allocated' => 'required|numeric|min:0',
            ]);

            if ($validator->fails()) {
                return redirect()->back()->with('error', $validator->errors()->first());
            }

            $stage = ConstructionStage::findOrFail($id);
            $stage->name = $request->name;
            $stage->description = $request->description;
            $stage->budget_allocated = $request->budget_allocated;
            $stage->start_date = $request->start_date;
            $stage->end_date = $request->end_date;
            $stage->status = $request->status;
            $stage->save();

            return redirect()->route('construction.stages.index', $stage->project_id)
                           ->with('success', __('Construction stage updated successfully.'));
        }
        return redirect()->back()->with('error', __('Permission Denied.'));
    }

    public function destroy($id)
    {
        if (Auth::user()->can('delete construction stage')) {
            $stage = ConstructionStage::findOrFail($id);
            $project_id = $stage->project_id;
            $stage->delete();

            return redirect()->route('construction.stages.index', $project_id)
                           ->with('success', __('Construction stage deleted successfully.'));
        }
        return redirect()->back()->with('error', __('Permission Denied.'));
    }

    public function updateOrder(Request $request)
    {
        if (Auth::user()->can('edit construction stage')) {
            $order = $request->order;
            foreach ($order as $index => $id) {
                ConstructionStage::where('id', $id)->update(['sequence_order' => $index]);
            }
            return response()->json(['success' => true]);
        }
        return response()->json(['error' => 'Permission Denied'], 403);
    }
}
```

#### 3.2 ConstructionActivityController

Similar structure to above, managing activities under stages.

---

### Phase 4: Create Blade Views

#### 4.1 Construction Stages Index

**File:** `resources/views/construction/stages/index.blade.php`
```blade
@extends('layouts.admin')

@section('page-title')
    {{ __('Construction Stages') }} - {{ $project->project_name }}
@endsection

@section('breadcrumb')
    <li class="breadcrumb-item"><a href="{{ route('dashboard') }}">{{ __('Dashboard') }}</a></li>
    <li class="breadcrumb-item"><a href="{{ route('projects.index') }}">{{ __('Projects') }}</a></li>
    <li class="breadcrumb-item"><a href="{{ route('projects.show', $project->id) }}">{{ $project->project_name }}</a></li>
    <li class="breadcrumb-item">{{ __('Construction Stages') }}</li>
@endsection

@section('action-btn')
    @can('create construction stage')
        <div class="float-end">
            <a href="#" data-url="{{ route('construction.stages.create', $project->id) }}"
               data-ajax-popup="true"
               data-title="{{ __('Create Construction Stage') }}"
               class="btn btn-sm btn-primary">
                <i class="ti ti-plus"></i> {{ __('Create') }}
            </a>
        </div>
    @endcan
@endsection

@push('script-page')
    <script src="{{ asset('assets/libs/jquery-ui/jquery-ui.js') }}"></script>
    <script>
        $(function () {
            $(".sortable").sortable({
                stop: function () {
                    var order = [];
                    $(this).find('li').each(function (index, data) {
                        order[index] = $(data).attr('data-id');
                    });
                    $.ajax({
                        url: "{{ route('construction.stages.order', $project->id) }}",
                        data: {order: order, _token: '{{ csrf_token() }}'},
                        type: 'POST',
                        success: function (data) {
                            show_toastr('{{ __("Success") }}', '{{ __("Order updated successfully") }}', 'success');
                        },
                        error: function (data) {
                            show_toastr('{{ __("Error") }}', '{{ __("Something went wrong") }}', 'error');
                        }
                    });
                }
            });
        });
    </script>
@endpush

@section('content')
    <div class="row">
        <div class="col-md-12">
            <div class="card">
                <div class="card-body">
                    <ul class="list-group sortable">
                        @forelse ($stages as $stage)
                            <li class="list-group-item" data-id="{{ $stage->id }}">
                                <div class="row align-items-center">
                                    <div class="col-4">
                                        <h6 class="mb-0">{{ $stage->name }}</h6>
                                        <small class="text-muted">
                                            <span class="badge badge-{{ ConstructionStage::$status_colors[$stage->status] }}">
                                                {{ ConstructionStage::$statuses[$stage->status] }}
                                            </span>
                                        </small>
                                    </div>
                                    <div class="col-2">
                                        <small class="text-muted">{{ __('Budget') }}</small>
                                        <div>₹{{ number_format($stage->budget_allocated, 2) }}</div>
                                    </div>
                                    <div class="col-2">
                                        <small class="text-muted">{{ __('Spent') }}</small>
                                        <div class="{{ $stage->getBudgetSpent() > $stage->budget_allocated ? 'text-danger' : '' }}">
                                            ₹{{ number_format($stage->getBudgetSpent(), 2) }}
                                        </div>
                                    </div>
                                    <div class="col-2">
                                        <small class="text-muted">{{ __('Progress') }}</small>
                                        <div class="progress mt-1" style="height: 8px;">
                                            <div class="progress-bar bg-success" role="progressbar"
                                                 style="width: {{ $stage->getProgressPercentage() }}%"></div>
                                        </div>
                                        <small>{{ $stage->getProgressPercentage() }}%</small>
                                    </div>
                                    <div class="col-2 text-end">
                                        <a href="{{ route('construction.activities.index', $stage->id) }}"
                                           class="btn btn-sm btn-info"
                                           title="{{ __('View Activities') }}">
                                            <i class="ti ti-list"></i>
                                        </a>
                                        @can('edit construction stage')
                                            <a href="#" data-url="{{ route('construction.stages.edit', $stage->id) }}"
                                               data-ajax-popup="true"
                                               data-title="{{ __('Edit Construction Stage') }}"
                                               class="btn btn-sm btn-primary">
                                                <i class="ti ti-pencil"></i>
                                            </a>
                                        @endcan
                                        @can('delete construction stage')
                                            {!! Form::open(['method' => 'DELETE', 'route' => ['construction.stages.destroy', $stage->id], 'class' => 'd-inline']) !!}
                                            <button type="submit" class="btn btn-sm btn-danger"
                                                    onclick="return confirm('{{ __('Are you sure?') }}')">
                                                <i class="ti ti-trash"></i>
                                            </button>
                                            {!! Form::close() !!}
                                        @endcan
                                    </div>
                                </div>
                            </li>
                        @empty
                            <li class="list-group-item text-center py-4">
                                <p class="text-muted">{{ __('No construction stages found. Click Create to add stages.') }}</p>
                            </li>
                        @endforelse
                    </ul>
                </div>
            </div>
        </div>
    </div>
@endsection
```

#### 4.2 Construction Stage Create Form

**File:** `resources/views/construction/stages/create.blade.php`
```blade
{{ Form::open(['route' => ['construction.stages.store', $project->id], 'method' => 'post']) }}
<div class="modal-body">
    <div class="row">
        <div class="form-group col-md-12">
            {{ Form::label('name', __('Stage Name'), ['class' => 'form-label']) }}
            {{ Form::text('name', null, ['class' => 'form-control', 'required' => 'required', 'placeholder' => __('e.g., Foundation Work')]) }}
        </div>

        <div class="form-group col-md-12">
            {{ Form::label('description', __('Description'), ['class' => 'form-label']) }}
            {{ Form::textarea('description', null, ['class' => 'form-control', 'rows' => 3, 'placeholder' => __('Describe this stage...')]) }}
        </div>

        <div class="form-group col-md-6">
            {{ Form::label('budget_allocated', __('Budget Allocated'), ['class' => 'form-label']) }}
            {{ Form::number('budget_allocated', null, ['class' => 'form-control', 'required' => 'required', 'step' => '0.01', 'min' => '0']) }}
        </div>

        <div class="form-group col-md-6">
            {{ Form::label('status', __('Status'), ['class' => 'form-label']) }}
            {{ Form::select('status', \App\Models\ConstructionStage::$statuses, 'pending', ['class' => 'form-control select']) }}
        </div>

        <div class="form-group col-md-6">
            {{ Form::label('start_date', __('Start Date'), ['class' => 'form-label']) }}
            {{ Form::date('start_date', null, ['class' => 'form-control']) }}
        </div>

        <div class="form-group col-md-6">
            {{ Form::label('end_date', __('End Date'), ['class' => 'form-label']) }}
            {{ Form::date('end_date', null, ['class' => 'form-control']) }}
        </div>
    </div>
</div>
<div class="modal-footer">
    <button type="button" class="btn btn-light" data-bs-dismiss="modal">{{ __('Cancel') }}</button>
    <button type="submit" class="btn btn-primary">{{ __('Create') }}</button>
</div>
{{ Form::close() }}
```

---

### Phase 5: Routes

**File:** `routes/web.php` (add these routes)
```php
Route::group(['middleware' => ['auth']], function () {
    // Construction Stages
    Route::get('projects/{project}/construction/stages', [ConstructionStageController::class, 'index'])
        ->name('construction.stages.index');
    Route::get('projects/{project}/construction/stages/create', [ConstructionStageController::class, 'create'])
        ->name('construction.stages.create');
    Route::post('projects/{project}/construction/stages', [ConstructionStageController::class, 'store'])
        ->name('construction.stages.store');
    Route::get('construction/stages/{stage}/edit', [ConstructionStageController::class, 'edit'])
        ->name('construction.stages.edit');
    Route::put('construction/stages/{stage}', [ConstructionStageController::class, 'update'])
        ->name('construction.stages.update');
    Route::delete('construction/stages/{stage}', [ConstructionStageController::class, 'destroy'])
        ->name('construction.stages.destroy');
    Route::post('projects/{project}/construction/stages/order', [ConstructionStageController::class, 'updateOrder'])
        ->name('construction.stages.order');

    // Construction Activities
    Route::get('construction/stages/{stage}/activities', [ConstructionActivityController::class, 'index'])
        ->name('construction.activities.index');
    Route::get('construction/stages/{stage}/activities/create', [ConstructionActivityController::class, 'create'])
        ->name('construction.activities.create');
    Route::post('construction/stages/{stage}/activities', [ConstructionActivityController::class, 'store'])
        ->name('construction.activities.store');
    Route::get('construction/activities/{activity}/edit', [ConstructionActivityController::class, 'edit'])
        ->name('construction.activities.edit');
    Route::put('construction/activities/{activity}', [ConstructionActivityController::class, 'update'])
        ->name('construction.activities.update');
    Route::delete('construction/activities/{activity}', [ConstructionActivityController::class, 'destroy'])
        ->name('construction.activities.destroy');
    Route::post('construction/activities/{activity}/status', [ConstructionActivityController::class, 'updateStatus'])
        ->name('construction.activities.updateStatus');
});
```

---

### Phase 6: Permissions

Add to permission seeder:

```php
$permissions = [
    'manage construction',
    'create construction stage',
    'edit construction stage',
    'delete construction stage',
    'view construction stage',
    'create construction activity',
    'edit construction activity',
    'delete construction activity',
    'view construction activity',
];
```

---

### Phase 7: Navigation Integration

Add to sidebar navigation (in layout partial):

```blade
@can('manage construction')
    <li class="dash-item">
        <a href="{{ route('projects.index') }}" class="dash-link">
            <span class="dash-micon"><i class="ti ti-building-factory"></i></span>
            <span class="dash-mtext">{{ __('Construction Projects') }}</span>
        </a>
    </li>
@endcan
```

Add tab to Project Show page:

```blade
<li class="nav-item">
    <a class="nav-link" data-bs-toggle="tab" href="#construction">
        <i class="ti ti-building"></i> {{ __('Construction Stages') }}
    </a>
</li>
```

---

## Summary

### What We're Building
- **New Tables**: `construction_stages` and `construction_activities`
- **New Models**: `ConstructionStage`, `ConstructionActivity`
- **New Controllers**: `ConstructionStageController`, `ConstructionActivityController`
- **New Views**: Stage and Activity management interfaces
- **Project Extension**: Add construction-specific methods to Project model

### Hierarchy
```
Project (existing)
  └── ConstructionStages (new)
       └── ConstructionActivities (new)
```

### Integration Points
1. Project show page gets new "Construction" tab
2. Construction stages link to project
3. Activities track costs and progress
4. Budget tracking at stage and project level
5. Activity assignment to users
6. Status workflows for stages and activities

---

**Next Steps:**
1. Run migrations
2. Create controllers
3. Create views
4. Test CRUD operations
5. Add permissions
6. Integrate with project page
7. Add reporting/dashboard widgets

---

**Document Version**: 1.0
**Created**: October 9, 2025
**Status**: Implementation Ready
